/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.pcode.emu.jit.gen.util;

import java.util.function.Consumer;

import org.objectweb.asm.MethodVisitor;

import ghidra.pcode.emu.jit.gen.util.Methods.Def;
import ghidra.pcode.emu.jit.gen.util.Methods.Def.ParamFunction;
import ghidra.pcode.emu.jit.gen.util.Methods.ObjDef;
import ghidra.pcode.emu.jit.gen.util.Types.*;

/**
 * The handle to a local variable
 * <p>
 * Direct use of the canonical constructor is not recommended. We may later hide this behind an
 * interface.
 * <p>
 * The JVM {@code <t>load} and {@code <t>store} instructions all take an index argument. We would
 * rather not have to keep track of indices, but instead wrap them in some named handle. These
 * handles are generated by {@link Scope#decl(BNonVoid, String)},
 * {@link Def#param(ParamFunction, BNonVoid, String, Consumer)}, and
 * {@link Def#done(ObjDef, TRef, Consumer)}. For the most part, the user need not worry at all about
 * indices, only types.
 * 
 * @param <T> the (machine) type of the variable.
 * @param type the type
 * @param name the name
 * @param index the index
 */
public record Local<T extends BNonVoid>(T type, String name, int index) {

	/**
	 * Construct a local variable handle
	 * <p>
	 * Direct use of this method is not recommended. It may be made private later.
	 * 
	 * @param <T> the type of the variable
	 * @param type the type
	 * @param name the name
	 * @param index the index
	 * @return the handle
	 */
	public static <T extends BNonVoid> Local<T> of(T type, String name, int index) {
		return new Local<>(type, name, index);
	}

	/**
	 * Declare a given local variable
	 * 
	 * @param <N> the stack contents of the emitter
	 * @param em the emitter (Nothing is actually emitted, but we need the wrapped ASM
	 *            {@link MethodVisitor}.)
	 * @param local the handle to the local
	 * @param start the start of the scope
	 * @param end the end of the scope
	 * @return the same emitter
	 */
	static <N> Emitter<N> decl(Emitter<N> em, Local<?> local, Lbl<?> start, Lbl<?> end) {
		em.mv.visitLocalVariable(local.name, local.type.type().getDescriptor(), null,
			start.label(), end.label(), local.index);
		return em;
	}
}
