/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package docking.widgets.search;

import docking.widgets.CursorPosition;
import docking.widgets.FindDialog;

/**
 * A simple interface for the {@link FindDialog} so that it can work for different search clients.
 * <p>
 * The {@link CursorPosition} object used by this interface is one that implementations can extend 
 * to add extra context to use when searching.  The implementation is responsible for creating the
 * locations and these locations will later be handed back to the searcher.
 * <p>
 * The {@link FindDialog} should use a single searcher for the life of the dialog.  This allows all
 * search results generated by that dialog to share the same worker queue for running background 
 * operations related to managing search results.
 */
public interface FindDialogSearcher {

	/**
	 * The current cursor position.  Used to search for the next item.
	 * @return the cursor position.
	 */
	public CursorPosition getCursorPosition();

	/**
	 * Returns the start cursor position.  This is used when a search is wrapped to start at the 
	 * beginning of the search range.
	 * @return the start position.
	 */
	public CursorPosition getStart();

	/**
	 * The end cursor position.  This is used when a search is wrapped while searching backwards to 
	 * start at the end position.
	 * @return the end position. 
	 */
	public CursorPosition getEnd();

	/**
	 * Perform a search for the next item in the given direction starting at the given cursor 
	 * position.
	 * @param text the search text.
	 * @param cursorPosition the current cursor position.
	 * @param searchForward true if searching forward.
	 * @param useRegex true if the search text is a regular expression; false if the text is literal.
	 * @return the search result or null if no match was found.
	 */
	public SearchResults search(String text, CursorPosition cursorPosition, boolean searchForward,
			boolean useRegex);

	/**
	 * Search for all matches.
	 * @param text the search text.
	 * @param useRegex true if the search text is a regular expression; false if the text is literal.
	 * @return all search results or an empty list.
	 */
	public SearchResults searchAll(String text, boolean useRegex);

	/**
	 * Disposes this searcher.  
	 */
	public void dispose();
}
