from __future__ import annotations
import collections.abc
import datetime
import typing
from warnings import deprecated # type: ignore

import jpype # type: ignore
import jpype.protocol # type: ignore

import java.awt # type: ignore
import java.lang # type: ignore
import java.nio # type: ignore
import java.nio.charset # type: ignore
import java.util # type: ignore
import java.util.regex # type: ignore


T = typing.TypeVar("T")


class AnsiColorResolver(java.lang.Object):
    """
    A mechanism for converting an ANSI color specification to an AWT color.
    """

    class WhichGround(java.lang.Enum[AnsiColorResolver.WhichGround]):
        """
        A stupid name for a thing that is either the foreground or the background.
        """

        class_: typing.ClassVar[java.lang.Class]
        FOREGROUND: typing.Final[AnsiColorResolver.WhichGround]
        BACKGROUND: typing.Final[AnsiColorResolver.WhichGround]

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> AnsiColorResolver.WhichGround:
            ...

        @staticmethod
        def values() -> jpype.JArray[AnsiColorResolver.WhichGround]:
            ...


    class ReverseVideo(java.lang.Enum[AnsiColorResolver.ReverseVideo]):

        class_: typing.ClassVar[java.lang.Class]
        NORMAL: typing.Final[AnsiColorResolver.ReverseVideo]
        REVERSED: typing.Final[AnsiColorResolver.ReverseVideo]

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> AnsiColorResolver.ReverseVideo:
            ...

        @staticmethod
        def values() -> jpype.JArray[AnsiColorResolver.ReverseVideo]:
            ...


    class_: typing.ClassVar[java.lang.Class]

    def resolveColor(self, color: VtHandler.AnsiColor, ground: AnsiColorResolver.WhichGround, intensity: VtHandler.Intensity, reverse: AnsiColorResolver.ReverseVideo) -> java.awt.Color:
        """
        Convert a color specification to an AWT color
        
        :param VtHandler.AnsiColor color: the ANSI color specification
        :param AnsiColorResolver.WhichGround ground: identifies the colors use in the foreground or the background
        :param VtHandler.Intensity intensity: gives the intensity of the color, really only used when a basic color is
                    specified.
        :param AnsiColorResolver.ReverseVideo reverse: identifies whether the foreground and background colors were swapped, really
                    only used when the default color is specified.
        :return: the AWT color, or null to not draw (usually in the case of the default background
                color)
        :rtype: java.awt.Color
        """


class VtAttributes(java.lang.Record):
    """
    A tuple of attributes to apply when rendering terminal text.
     
     
    
    These are set and collected as the parser and handler deal with various ANSI VT escape codes. As
    characters are placed in the buffer, the current attributes are applied to the corresponding
    cells. The renderer then has to apply the attributes appropriately as it renders each character
    in the buffer.
    """

    class_: typing.ClassVar[java.lang.Class]
    DEFAULTS: typing.Final[VtAttributes]
    """
    The default attributes: plain white on black, usually.
    """


    def __init__(self, fg: VtHandler.AnsiColor, bg: VtHandler.AnsiColor, intensity: VtHandler.Intensity, font: VtHandler.AnsiFont, underline: VtHandler.Underline, blink: VtHandler.Blink, reverse: AnsiColorResolver.ReverseVideo, hidden: typing.Union[jpype.JBoolean, bool], strikeThrough: typing.Union[jpype.JBoolean, bool], proportionalSpacing: typing.Union[jpype.JBoolean, bool]):
        ...

    @typing.overload
    def bg(self, bg: VtHandler.AnsiColor) -> VtAttributes:
        """
        Create a copy of this record with the background color replaced
        
        :param VtHandler.AnsiColor bg: the new background color
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def bg(self) -> VtHandler.AnsiColor:
        ...

    @typing.overload
    def blink(self, blink: VtHandler.Blink) -> VtAttributes:
        """
        Create a copy of this record with the blink replaced
        
        :param VtHandler.Blink blink: the new blink
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def blink(self) -> VtHandler.Blink:
        ...

    def equals(self, o: java.lang.Object) -> bool:
        ...

    @typing.overload
    def fg(self, fg: VtHandler.AnsiColor) -> VtAttributes:
        """
        Create a copy of this record with the foreground color replaced
        
        :param VtHandler.AnsiColor fg: the new foreground color
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def fg(self) -> VtHandler.AnsiColor:
        ...

    @typing.overload
    def font(self, font: VtHandler.AnsiFont) -> VtAttributes:
        """
        Create a copy of this record with the font replaced
        
        :param VtHandler.AnsiFont font: the new font
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def font(self) -> VtHandler.AnsiFont:
        ...

    def hashCode(self) -> int:
        ...

    @typing.overload
    def hidden(self, hidden: typing.Union[jpype.JBoolean, bool]) -> VtAttributes:
        """
        Create a copy of this record with the hidden replaced
        
        :param jpype.JBoolean or bool hidden: the new hidden
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def hidden(self) -> bool:
        ...

    @typing.overload
    def intensity(self, intensity: VtHandler.Intensity) -> VtAttributes:
        """
        Create a copy of this record with the intensity replaced
        
        :param VtHandler.Intensity intensity: the new intensity
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def intensity(self) -> VtHandler.Intensity:
        ...

    @typing.overload
    def proportionalSpacing(self, proportionalSpacing: typing.Union[jpype.JBoolean, bool]) -> VtAttributes:
        """
        Create a copy of this record with the proportional-spacing replaced
        
        :param jpype.JBoolean or bool proportionalSpacing: the new proportional-spacing
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def proportionalSpacing(self) -> bool:
        ...

    def resolveBackground(self, colors: AnsiColorResolver) -> java.awt.Color:
        """
        Resolve the background color for these attributes
        
        :param AnsiColorResolver colors: the color resolver
        :return: the color, or null to not paint the background
        :rtype: java.awt.Color
        """

    def resolveForeground(self, colors: AnsiColorResolver) -> java.awt.Color:
        """
        Resolve the foreground color for these attributes
        
        :param AnsiColorResolver colors: the color resolver
        :return: the color
        :rtype: java.awt.Color
        """

    def reverse(self) -> AnsiColorResolver.ReverseVideo:
        ...

    def reverseVideo(self, reverse: AnsiColorResolver.ReverseVideo) -> VtAttributes:
        """
        Create a copy of this record with the reverse-video replaced
        
        :param AnsiColorResolver.ReverseVideo reverse: the new reverse-video
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def strikeThrough(self, strikeThrough: typing.Union[jpype.JBoolean, bool]) -> VtAttributes:
        """
        Create a copy of this record with the strike-through replaced
        
        :param jpype.JBoolean or bool strikeThrough: the new strike-through
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def strikeThrough(self) -> bool:
        ...

    def toString(self) -> str:
        ...

    @typing.overload
    def underline(self, underline: VtHandler.Underline) -> VtAttributes:
        """
        Create a copy of this record with the underline replaced
        
        :param VtHandler.Underline underline: the new underline
        :return: the new record
        :rtype: VtAttributes
        """

    @typing.overload
    def underline(self) -> VtHandler.Underline:
        ...


class VtBuffer(java.lang.Object):
    """
    A buffer for a terminal display and scroll-back
     
     
    
    This object implements all of the buffer, line, and character manipulations available in the
    terminal. It's likely more will need to be added in the future. While the ANSI VT parser
    determines what commands to execute, this buffer provides the actual implementation of those
    commands.
    """

    class LineConsumer(java.lang.Object):
        """
        A callback for iterating over the lines of the buffer
        """

        class_: typing.ClassVar[java.lang.Class]

        def accept(self, i: typing.Union[jpype.JInt, int], y: typing.Union[jpype.JInt, int], t: VtLine):
            """
            Process a line of terminal text
            
            :param jpype.JInt or int i: the index of the line, optionally including scroll-back, 0 up, top to bottom
            :param jpype.JInt or int y: the vertical position of the line. For a scroll-back line, this is -1.
                        Otherwise, this counts 0 up, top to bottom.
            :param VtLine t: the line
            
            .. seealso::
            
                | :obj:`VtBuffer.forEachLine(boolean, LineConsumer)`
            """


    class_: typing.ClassVar[java.lang.Class]
    DEFAULT_ROWS: typing.Final = 25
    DEFAULT_COLS: typing.Final = 80

    @typing.overload
    def __init__(self):
        """
        Create a new buffer of the default size (25 lines, 80 columns)
        """

    @typing.overload
    def __init__(self, rows: typing.Union[jpype.JInt, int], cols: typing.Union[jpype.JInt, int]):
        """
        Create a new buffer of the given size
        
        :param jpype.JInt or int rows: the number of rows
        :param jpype.JInt or int cols: the number of columns
        """

    def carriageReturn(self):
        """
        Move the cursor back to the beginning of the line
         
         
        
        This does *not* move the cursor down.
        """

    def checkVerticalScroll(self):
        """
        If the cursor is beyond the bottom of the display, scroll the viewport down and move the
        cursor up until the cursor is at the bottom of the display. If applicable, lines at the top
        of the display is shifted into the scroll-back buffer.
        """

    def deleteChars(self, n: typing.Union[jpype.JInt, int]):
        """
        Delete n characters at the cursor
         
         
        
        Characters at (and ``n-1`` to the right) of the cursor are deleted. The remaining
        characters to the right are shifted left ``n`` columns.
        
        :param jpype.JInt or int n: the number of characters to delete
        """

    def deleteLines(self, n: typing.Union[jpype.JInt, int]):
        """
        Delete n lines at the cursor
         
         
        
        Lines at (and immediately below) the cursor are removed and all lines between the cursor and
        the bottom of the viewport are shifted up to make room for n blank lines inserted at (and
        above) the bottom of the viewport. None of the lines above the cursor are affected.
        
        :param jpype.JInt or int n: the number of lines to delete
        """

    def erase(self, erasure: VtHandler.Erasure):
        """
        Erase (clear) some portion of the display buffer
         
         
        
        If the current line is erased from start to the cursor, the cursor's attributes are applied
        to the cleared columns.
        
        :param VtHandler.Erasure erasure: specifies what, relative to the cursor, to erase.
        """

    def eraseChars(self, n: typing.Union[jpype.JInt, int]):
        """
        Erase n characters at the cursor
         
         
        
        Characters at (and ``n-1`` to the right) of the cursor are erased, i.e., replaced with
        blanks. No shifting takes place.
        
        :param jpype.JInt or int n: the number of characters to erase
        """

    def forEachLine(self, includeScrollBack: typing.Union[jpype.JBoolean, bool], action: VtBuffer.LineConsumer):
        """
        Perform an action on each line of terminal text, optionally including the scroll-back buffer.
        
        :param jpype.JBoolean or bool includeScrollBack: true to include the scroll-back buffer
        :param VtBuffer.LineConsumer action: the action
        """

    def getAttributes(self) -> VtAttributes:
        """
        Get the cursor's current attributes
         
         
        
        Characters put into the buffer via :meth:`putChar(char) <.putChar>` are assigned the cursor's current
        attributes at the time they are inserted.
        
        :return: the current attributes
        :rtype: VtAttributes
        
        .. seealso::
        
            | :obj:`.setAttributes(VtAttributes)`
        """

    def getCols(self) -> int:
        """
        Get the number of columns in the display
         
         
        
        This is not just the number of columns currently being used. This is the "columns" dimension
        of the display, i.e., the maximum number of characters in a rows before wrapping.
        
        :return: the number of columns
        :rtype: int
        """

    def getCurX(self) -> int:
        """
        Get the cursor's column, 0 up, left to right
        
        :return: the column
        :rtype: int
        """

    def getCurY(self) -> int:
        """
        Get the cursor's row, 0 up, top to bottom
        
        :return: the row
        :rtype: int
        """

    def getRows(self) -> int:
        """
        Get the number of rows in the display
         
         
        
        This is not just the number of rows currently being used. This is the "rows" dimension of the
        display, i.e., the maximum number of rows it can display before scrolling.
        
        :return: the number of rows
        :rtype: int
        """

    def getScrollBackSize(self) -> int:
        """
        Get the number of lines in the scroll-back buffer
        
        :return: the number of lines
        :rtype: int
        """

    def getText(self, startRow: typing.Union[jpype.JInt, int], startCol: typing.Union[jpype.JInt, int], endRow: typing.Union[jpype.JInt, int], endCol: typing.Union[jpype.JInt, int], lineSep: java.lang.CharSequence) -> str:
        """
        Get the text between two locations in the buffer
         
         
        
        The buffer attempts to avoid extraneous space at the end of each line. This isn't always
        perfect and depends on how lines are cleared. If they are cleared using
        :meth:`erase(Erasure) <.erase>`, then the buffer will cull the trailing spaces resulting from the
        clear. If they are cleared using :meth:`putChar(char) <.putChar>` passing a space ``' '``, then the
        inserted spaces will be included. In practice, this depends on the application controlling
        the terminal.
         
         
        
        Like the other methods, locations are specified 0 up, top to bottom, and left to right.
        Unlike the other methods, the ending character is excluded from the result.
        
        :param jpype.JInt or int startRow: the row for the starting location, inclusive
        :param jpype.JInt or int startCol: the column for the starting location, inclusive
        :param jpype.JInt or int endRow: the row for the ending location, inclusive
        :param jpype.JInt or int endCol: the column for the ending location, *exclusive*
        :param java.lang.CharSequence lineSep: the line separator
        :return: the text
        :rtype: str
        """

    def insertChars(self, n: typing.Union[jpype.JInt, int]):
        """
        Insert n blank characters at the cursor
         
         
        
        Any characters right the cursor on the same line are shifted right to make room and n blanks
        are inserted at (and to the right) of the cursor. No wrapping occurs. Characters that would
        be moved or inserted right of the display buffer are effectively deleted. The cursor is
        *not* moved after this operation.
        
        :param jpype.JInt or int n: the number of blanks to insert.
        """

    def insertLines(self, n: typing.Union[jpype.JInt, int]):
        """
        Insert n blank lines at the cursor
         
         
        
        Lines at the bottom of the viewport are removed and all the lines between the cursor and the
        bottom of the viewport are shifted down, to make room for n blank lines. None of the lines
        above the cursor are affected, including those in the scroll-back buffer.
        
        :param jpype.JInt or int n: the number of lines to insert
        """

    def moveCursor(self, row: typing.Union[jpype.JInt, int], col: typing.Union[jpype.JInt, int]):
        """
        Move the cursor to the given row and column
        
         
        
        The position is clamped to the dimensions of the display. No scrolling will take place if
        ``col`` exceeds the number of rows.
        
        :param jpype.JInt or int row: the desired row, 0 up, top to bottom
        :param jpype.JInt or int col: the desired column, 0 up, left to right
        """

    def moveCursorDown(self, n: typing.Union[jpype.JInt, int], dedupWrap: typing.Union[jpype.JBoolean, bool]):
        """
        Move the cursor down n rows
         
         
        
        If the cursor would move below the bottom of the display, the viewport will be scrolled so
        that the cursor remains in the display. The value of n must be positive, otherwise behavior
        is undefined. To move the cursor up, use :meth:`moveCursorUp(int) <.moveCursorUp>`.
         
         
        
        ConPty has a habit of moving the cursor past the end of the current line before sending CRLF.
        (Though, I imagine there are other applications that might do this.) The ``dedupWrap``
        parameter is made to accommodate this. If it is set, n is a single line, and the previous
        line was wrapped, then this does nothing more than remove the wrapped flag from the previous
        line.
        
        :param jpype.JInt or int n: the number of lines to move down
        :param jpype.JBoolean or bool dedupWrap: whether to detect and ignore a line feed after wrapping
        """

    def moveCursorLeft(self, n: typing.Union[jpype.JInt, int], wrap: typing.Union[jpype.JBoolean, bool]):
        """
        Move the cursor left (backward) n columns
         
         
        
        The cursor is clamped into the display. If wrap is specified, the cursor would exceed the
        left side of the display, and the previous line was wrapped onto the current line, then the
        cursor will instead be moved to the end of the previous line. (It doesn't matter how far the
        cursor would exceed the left; it moves up at most one line.) The value of n must be positive,
        otherwise behavior is undefined. To move the cursor right, use :meth:`moveCursorRight(int) <.moveCursorRight>`.
        
        :param jpype.JInt or int n: the number of columns
        :param jpype.JBoolean or bool wrap: whether to wrap the cursor to the previous line if would exceed the left of the
                    display
        """

    def moveCursorRight(self, n: typing.Union[jpype.JInt, int], wrap: typing.Union[jpype.JBoolean, bool], isCursorShowing: typing.Union[jpype.JBoolean, bool]):
        """
        Move the cursor right (forward) n columns
         
         
        
        The cursor is clamped into the display. If wrap is specified and the cursor would exceed the
        right side of the display, the cursor will instead be wrapped to the start of the next line,
        possibly scrolling the viewport down. (It doesn't matter how far the cursor exceeds the
        right; the cursor moves down exactly one line.) The value of n must be positive, otherwise
        behavior is undefined. To move the cursor left, use :meth:`moveCursorLeft(int) <.moveCursorLeft>`.
        
        :param jpype.JInt or int n: the number of columns
        :param jpype.JBoolean or bool wrap: whether to wrap the cursor to the next line if it would exceed the right of the
                    display
        """

    def moveCursorUp(self, n: typing.Union[jpype.JInt, int]):
        """
        Move the cursor up n rows
         
         
        
        The cursor cannot move above the top of the display. The value of n must be positive,
        otherwise behavior is undefined. To move the cursor down, use :meth:`moveCursorDown(int) <.moveCursorDown>`.
        
        :param jpype.JInt or int n: the number of rows to move the cursor up
        """

    def putChar(self, c: typing.Union[jpype.JChar, int, str]):
        """
        Put the given character at the cursor, and move the cursor forward
         
         
        
        The cursor's current attributes are applied to the character.
        
        :param jpype.JChar or int or str c: the character to put into the buffer
        
        .. seealso::
        
            | :obj:`.setAttributes(VtAttributes)`
        
            | :obj:`.getAttributes()`
        """

    def reset(self):
        """
        Clear the buffer and all state, as if it has just been created
        """

    def resetBottomY(self) -> int:
        ...

    def resize(self, cols: typing.Union[jpype.JInt, int], rows: typing.Union[jpype.JInt, int]) -> bool:
        """
        Resize the buffer to the given number of rows and columns
         
         
        
        The viewport is reset to include the full display. Each line, including those in the
        scroll-back buffer are resized to match the requested number of columns. If the row count is
        decreasing, lines at the top of the display are be shifted into the scroll-back buffer. If
        the row count is increasing, lines at the bottom of the scroll-back buffer are shifted into
        the display buffer. The scroll-back buffer may be culled if the resulting number of lines
        exceeds that scroll-back maximum. The cursor position is adjusted so that, if possible, it
        remains on the same line. (The cursor cannot enter the scroll-back region.) Finally, the
        cursor is clamped into the display region. The saved cursor, if applicable, is similarly
        treated.
        
        :param jpype.JInt or int cols: the number of columns
        :param jpype.JInt or int rows: the number of rows
        :return: true if the buffer was actually resized
        :rtype: bool
        """

    def restoreCursorPos(self):
        """
        Restore a saved cursor position
         
         
        
        If there was no previous call to :meth:`saveCursorPos() <.saveCursorPos>`, the cursor is placed at the
        top-left of the display.
        """

    def saveCursorPos(self):
        """
        Save the current cursor position
         
         
        
        There is only one slot for the saved cursor. It is not a stack or anything fancy. To restore
        the cursor, use :meth:`restoreCursorPos() <.restoreCursorPos>`. The advantage to using this vice
        :meth:`getCurX() <.getCurX>` and :meth:`getCurY() <.getCurY>` to save it externally, is that the buffer will
        adjust the saved position if the buffer is resized via :meth:`resize(int, int) <.resize>`.
        """

    def scrollViewportDown(self, intoScrollBack: typing.Union[jpype.JBoolean, bool]):
        """
        Scroll the viewport down a line
         
         
        
        The lines are shifted upward. The line at the top of the viewport is removed, and a blank
        line is inserted at the bottom of the viewport. If the viewport includes the display's top
        line and intoScrollBack is specified, the line is shifted into the scroll-back buffer.
        """

    def scrollViewportUp(self):
        """
        Scroll the viewport up a line
         
         
        
        The lines are shifted downward. The line at the bottom of the viewport is removed, and a
        blank line is inserted at the top of the viewport.
        """

    def setAttributes(self, attributes: VtAttributes):
        """
        Set the cursor's current attributes
         
         
        
        These are usually the attributes given by the ANSI SGR control sequences. They may not affect
        the display of the cursor itself, but rather of the characters placed at the cursor via
        :meth:`putChar(char) <.putChar>`. NOTE: Not all attributes are necessarily supported by the renderer.
        
        :param VtAttributes attributes: the desired attributes
        """

    def setMaxScrollBack(self, maxScrollBack: typing.Union[jpype.JInt, int]):
        """
        Adjust the maximum number of lines in the scroll-back buffer
         
         
        
        If the scroll-back buffer exceeds the given maximum, it is immediately culled.
        
        :param jpype.JInt or int maxScrollBack: the maximum number of scroll-back lines
        """

    def setScrollViewport(self, start: typing.Union[java.lang.Integer, int], end: typing.Union[java.lang.Integer, int]):
        """
        Specify the scrolling viewport of the buffer
         
         
        
        By default, the viewport is the entire display, and scrolling the viewport downward may cause
        lines to enter the scroll-back buffer. The buffer manages these boundaries so that they can
        be updated on calls to :meth:`resize(int, int) <.resize>`. Both parameters are optional, though
        ``end`` should likely only be given if ``start`` is also given. The parameters are
        silently adjusted to ensure that both are within the bounds of the display and so that the
        end is at or below the start. Once set, the cursor should remain within the viewport, or
        otherwise cause the viewport to scroll. Operations that would cause the display to scroll,
        instead cause just the viewport to scroll. Additionally, cursor movement operations are
        clamped to the viewport.
        
        :param java.lang.Integer or int start: the first line in the viewport, 0 up, top to bottom, inclusive. If omitted, this
                    is the top line of the display.
        :param java.lang.Integer or int end: the last line in the viewport, 0 up, top to bottom, inclusive. If omitted, this is
                    the bottom line of the display.
        """

    def size(self) -> int:
        """
        Get the total number of lines, including scroll-back lines, in the buffer
         
         
        
        This is equal to :meth:`getScrollBackSize() <.getScrollBackSize>```+``:meth:`getRows() <.getRows>`.
        
        :return: the number of lines
        :rtype: int
        """

    def tab(self):
        """
        More the cursor forward to the next tab stop
        """

    def tabBack(self):
        """
        Move the cursor backward to the previous tab stop
        """

    @property
    def scrollBackSize(self) -> jpype.JInt:
        ...

    @property
    def curX(self) -> jpype.JInt:
        ...

    @property
    def curY(self) -> jpype.JInt:
        ...

    @property
    def attributes(self) -> VtAttributes:
        ...

    @attributes.setter
    def attributes(self, value: VtAttributes):
        ...

    @property
    def rows(self) -> jpype.JInt:
        ...

    @property
    def cols(self) -> jpype.JInt:
        ...


class VtCharset(java.lang.Enum[VtCharset]):
    """
    A legacy style charset
     
     
    
    Finding the particulars for these online has not been fun, so these are implemented on an
    as-needed basis. There's probably a simple translation to some unicode code pages, since those
    seem to be ordered by some of these legacy character sets. The default implementation for each
    charset will just be equivalent to US-ASCII. There's a lot of plumbing missing around these, two.
    For example, I'm assuming that switching to "the alternate charset" means using G1 instead of G0.
    I've not read carefully enough to know how G2 or G3 are used.
     
     
    
    It'd be nice to just use UTF-8, but the application would have to agree.
    """

    class G(java.lang.Enum[VtCharset.G]):
        """
        The designation for a charset slot
         
         
        
        It seems the terminal allows for the selection of 4 alternative charsets, the first of which
        G0 is the default or primary.
        """

        class_: typing.ClassVar[java.lang.Class]
        G0: typing.Final[VtCharset.G]
        G1: typing.Final[VtCharset.G]
        G2: typing.Final[VtCharset.G]
        G3: typing.Final[VtCharset.G]
        b: typing.Final[jpype.JByte]

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtCharset.G:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtCharset.G]:
            ...


    class_: typing.ClassVar[java.lang.Class]
    UK: typing.Final[VtCharset]
    USASCII: typing.Final[VtCharset]
    FINNISH: typing.Final[VtCharset]
    SWEDISH: typing.Final[VtCharset]
    GERMAN: typing.Final[VtCharset]
    FRENCH_CANADIAN: typing.Final[VtCharset]
    FRENCH: typing.Final[VtCharset]
    ITALIAN: typing.Final[VtCharset]
    SPANISH: typing.Final[VtCharset]
    DUTCH: typing.Final[VtCharset]
    GREEK: typing.Final[VtCharset]
    TURKISH: typing.Final[VtCharset]
    PORTUGESE: typing.Final[VtCharset]
    HEBREW: typing.Final[VtCharset]
    SWISS: typing.Final[VtCharset]
    NORWEGIAN_DANISH: typing.Final[VtCharset]
    DEC_SPECIAL_LINES: typing.Final[VtCharset]
    DEC_SUPPLEMENTAL: typing.Final[VtCharset]
    DEC_TECHNICAL: typing.Final[VtCharset]
    DEC_HEBREW: typing.Final[VtCharset]
    DEC_GREEK: typing.Final[VtCharset]
    DEC_TURKISH: typing.Final[VtCharset]
    DEC_SUPPLEMENTAL_GRAPHICS: typing.Final[VtCharset]
    DEC_CYRILLIC: typing.Final[VtCharset]

    def mapChar(self, c: typing.Union[jpype.JChar, int, str]) -> str:
        """
        Map a character, as decoded using US-ASCII, into the actual character for the character set.
        
        :param jpype.JChar or int or str c: the character from US-ASCII.
        :return: the mapped character
        :rtype: str
        """

    @staticmethod
    def valueOf(name: typing.Union[java.lang.String, str]) -> VtCharset:
        ...

    @staticmethod
    def values() -> jpype.JArray[VtCharset]:
        ...


class VtHandler(java.lang.Object):
    """
    The handler of parsed ANSI VT control sequences
     
     
    
    Here are some of the resources where I found useful documentation:
     
     
    * XTerm Control
    Sequences
    * ANSI/VT100
    Terminal Control Escape Sequences
    * Wikipedia: ANSI escape code
    
     
     
    
    They were incredibly useful, even when experimentation was required to fill in details, because
    they at least described the sort of behavior I should be looking for. Throughout the referenced
    documents and within this documentation, the following abbreviations are used for escape
    sequences:
     
     
    +--------------+-----------+--------------+
    | Abbreviation | Sequence  | Java String  |
    +==============+===========+==============+
    |``ESC``       |byte 0x1b  |``"\033"``    |
    +--------------+-----------+--------------+
    |``CSI``       |``ESC [``  |``"\033["``   |
    +--------------+-----------+--------------+
    |``OSC``       |``ESC ]``  |``"\033]"``   |
    +--------------+-----------+--------------+
    |``ST``        |``ESC \``  |``"\033\\"``  |
    +--------------+-----------+--------------+
    |``BEL``       |byte 0x07  |``"\007"``    |
    +--------------+-----------+--------------+
    
     
     
    
    The separation between the parser and the handler deals in state management. The parser manages
    state only of the control sequence parser itself, i.e., the current node in the token parsing
    state machine. The state of the terminal, e.g., the current attributes, cursor position, etc.,
    are managed by the handler and its delegates.
     
     
    
    For example, the Cursor Position sequence is documented as:
     
    
    ``CSI <em>n</em> ; <em>m</em> H``
     
    
    Supposing ``n`` is 13 and ``m`` is 40, this sequence would be encoded as the string
    ``"\033[13;40H"``. The parser will handle decoding the CSI, parameters, and final byte
    ``'H'``. It will then invoke :meth:`handleCsi(ByteBuffer, ByteBuffer, byte) <.handleCsi>`. The default
    implementation provided by this interface handles many of the final bytes, including ``'H'``.
    It will thus invoke the abstract :meth:`handleMoveCursor(int, int) <.handleMoveCursor>` method passing 12 and 39.
    Note that 1 is subtracted from both parameters, because ANSI specifies 1-up indexing while Java
    lends itself to 0-up indexing.
     
     
    
    The XTerm documentation, which is arguably the most thorough, presents the CSI commands
    alphabetically by the final byte, in ASCII order. For sanity and consistency, we adopt the same
    ordering in our switch cases.
    """

    class AnsiColor(java.lang.Object):
        """
        An ANSI color specification
        
         
        
        We avoid going straight to AWT colors, 1) Because it provides better separation between the
        terminal logic and the rendering framework, and 2) Because some specifications, e.g., default
        background, are better delayed until the renderer has gathered the necessary context to
        resolve it. Various enums and records implement this interface to provide the specifcations.
        """

        class_: typing.ClassVar[java.lang.Class]


    class AnsiDefaultColor(java.lang.Enum[VtHandler.AnsiDefaultColor], VtHandler.AnsiColor):
        """
        A singleton representing the default color
        
         
        
        The actual color selected will depend on context and use. Most notably, the default color
        used for foreground should greatly contrast the default color used for the background.
        """

        class_: typing.ClassVar[java.lang.Class]
        INSTANCE: typing.Final[VtHandler.AnsiDefaultColor]

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.AnsiDefaultColor:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.AnsiDefaultColor]:
            ...


    class AnsiStandardColor(java.lang.Enum[VtHandler.AnsiStandardColor], VtHandler.AnsiColor):
        """
        One of the eight standard ANSI colors
        
         
        
        The actual color may be modified by other SGR attributes, notably :obj:`Intensity`. For
        colors that are described by hue, some thought should be given to how the standard and
        intense versions differ. Some palettes may choose a darker color, reserving the brightest for
        the intense version. Others may use the brightest, choosing to whiten the intense version.
        """

        class_: typing.ClassVar[java.lang.Class]
        BLACK: typing.Final[VtHandler.AnsiStandardColor]
        """
        Usually the darkest black available. Implementations may select a color softer on the
        eyes, depending on use. For foreground, this should likely be true black (0,0,0).
        """

        RED: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly red.
        """

        GREEN: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly green.
        """

        YELLOW: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly yellow.
        """

        BLUE: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly blue. For palettes made to display on a dark (but not black)
        background, a hue tinted toward cyan is recommended.
        """

        MAGENTA: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly magenta or purple. For palettes made to display on a dark
        (but not black) background, a hue tinted toward red is recommended.
        """

        CYAN: typing.Final[VtHandler.AnsiStandardColor]
        """
        A color whose hue is clearly cyan.
        """

        WHITE: typing.Final[VtHandler.AnsiStandardColor]
        """
        A relatively bright white, sparing the brightest for intense white.
        """

        ALL: typing.Final[java.util.List[VtHandler.AnsiStandardColor]]
        """
        An unmodifiable list giving all the standard colors
        """

        intense: typing.Final[VtHandler.AnsiIntenseColor]
        dim: typing.Final[VtHandler.AnsiDimColor]

        @staticmethod
        def get(code: typing.Union[jpype.JInt, int]) -> VtHandler.AnsiStandardColor:
            """
            Get the standard color for the given numerical code
             
             
            
            For example, the sequence ``CSI [ 34 m`` would use code 4 (blue).
            
            :param jpype.JInt or int code: the code
            :return: the color
            :rtype: VtHandler.AnsiStandardColor
            """

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.AnsiStandardColor:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.AnsiStandardColor]:
            ...


    class AnsiIntenseColor(java.lang.Enum[VtHandler.AnsiIntenseColor], VtHandler.AnsiColor):
        """
        One of the eight ANSI intense colors
         
         
        
        Note that intense colors may also be specified using the standard color with the
        :obj:`Intensity.BOLD` attribute, depending on the command sequence.
        """

        class_: typing.ClassVar[java.lang.Class]
        BLACK: typing.Final[VtHandler.AnsiIntenseColor]
        """
        A relatively dark grey, but not true black.
        """

        RED: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.RED`, but brighter and/or whiter.
        """

        GREEN: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.GREEN`, but brighter and/or whiter.
        """

        YELLOW: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.YELLOW`, but brighter and/or whiter.
        """

        BLUE: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.BLUE`, but brighter and/or whiter.
        """

        MAGENTA: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.MAGENTA`, but brighter and/or whiter.
        """

        CYAN: typing.Final[VtHandler.AnsiIntenseColor]
        """
        See :obj:`AnsiStandardColor.CYAN`, but brighter and/or whiter.
        """

        WHITE: typing.Final[VtHandler.AnsiIntenseColor]
        """
        Usually the brightest white available.
        """

        ALL: typing.Final[java.util.List[VtHandler.AnsiIntenseColor]]
        """
        An unmodifiable list giving all the intense colors
        """


        @staticmethod
        def get(code: typing.Union[jpype.JInt, int]) -> VtHandler.AnsiIntenseColor:
            """
            Get the intense color for the given numerical code
             
             
            
            For example, the sequence ``CSI [ 94 m`` would use code 4 (blue).
            
            :param jpype.JInt or int code: the code
            :return: the color
            :rtype: VtHandler.AnsiIntenseColor
            """

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.AnsiIntenseColor:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.AnsiIntenseColor]:
            ...


    class AnsiDimColor(java.lang.Enum[VtHandler.AnsiDimColor], VtHandler.AnsiColor):
        """
        One of the eight ANSI dim colors
        """

        class_: typing.ClassVar[java.lang.Class]
        BLACK: typing.Final[VtHandler.AnsiDimColor]
        """
        A relatively dark grey, but not true black.
        """

        RED: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.RED`, but darker.
        """

        GREEN: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.GREEN`, but darker.
        """

        YELLOW: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.YELLOW`, but darker.
        """

        BLUE: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.BLUE`, but darker.
        """

        MAGENTA: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.MAGENTA`, but darker.
        """

        CYAN: typing.Final[VtHandler.AnsiDimColor]
        """
        See :obj:`AnsiStandardColor.CYAN`, but darker.
        """

        WHITE: typing.Final[VtHandler.AnsiDimColor]
        """
        Usually grey.
        """

        ALL: typing.Final[java.util.List[VtHandler.AnsiDimColor]]
        """
        An unmodifiable list giving all the dim colors
        """


        @staticmethod
        def get(code: typing.Union[jpype.JInt, int]) -> VtHandler.AnsiDimColor:
            """
            Get the dim color for the given numerical code
             
             
            
            For example, the sequence ``CSI [ 34 m`` would use code 4 (blue).
            
            :param jpype.JInt or int code: the code
            :return: the color
            :rtype: VtHandler.AnsiDimColor
            """

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.AnsiDimColor:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.AnsiDimColor]:
            ...


    class Ansi216Color(java.lang.Record, VtHandler.AnsiColor):
        """
        For 8-bit colors, one of the 216 colors from the RGB cube
         
         
        
        The r, g, and b fields give the "step" number from 0 to 5, dimmest to brightest.
        """

        class_: typing.ClassVar[java.lang.Class]

        def __init__(self, r: typing.Union[jpype.JInt, int], g: typing.Union[jpype.JInt, int], b: typing.Union[jpype.JInt, int]):
            ...

        def b(self) -> int:
            ...

        def equals(self, o: java.lang.Object) -> bool:
            ...

        def g(self) -> int:
            ...

        def hashCode(self) -> int:
            ...

        def r(self) -> int:
            ...

        def toString(self) -> str:
            ...


    class AnsiGrayscaleColor(java.lang.Record, VtHandler.AnsiColor):
        """
        For 8-bit colors, one of the 24 grays
         
         
        
        The v field is a value from 0 to 23, 0 being the dimmest, but not true black, and 23 being
        the brightest, but not true white.
        """

        class_: typing.ClassVar[java.lang.Class]

        def __init__(self, v: typing.Union[jpype.JInt, int]):
            ...

        def equals(self, o: java.lang.Object) -> bool:
            ...

        def hashCode(self) -> int:
            ...

        def toString(self) -> str:
            ...

        def v(self) -> int:
            ...


    class Ansi24BitColor(java.lang.Record, VtHandler.AnsiColor):
        """
        A 24-bit color
         
         
        
        The r, g, and b fields are values from 0 to 255 dimmest to brightest.
        """

        class_: typing.ClassVar[java.lang.Class]

        def __init__(self, r: typing.Union[jpype.JInt, int], g: typing.Union[jpype.JInt, int], b: typing.Union[jpype.JInt, int]):
            ...

        def b(self) -> int:
            ...

        def equals(self, o: java.lang.Object) -> bool:
            ...

        def g(self) -> int:
            ...

        def hashCode(self) -> int:
            ...

        def r(self) -> int:
            ...

        def toString(self) -> str:
            ...


    class Intensity(java.lang.Enum[VtHandler.Intensity]):
        """
        Modifies the intensity of the character either by color or by font weight.
         
         
        
        The renderer may choose a combination of strategies. For example, :obj:`.NORMAL` may be
        rendered using the standard color and bold type. Then :obj:`.BOLD` would use the intense
        color, keeping the bold type; whereas :obj:`.DIM` would use normal type, keeping the
        standard color. Some user configuration may be desired here.
        """

        class_: typing.ClassVar[java.lang.Class]
        NORMAL: typing.Final[VtHandler.Intensity]
        """
        The default intensity
        """

        BOLD: typing.Final[VtHandler.Intensity]
        """
        More intense than :obj:`.NORMAL`
        """

        DIM: typing.Final[VtHandler.Intensity]
        """
        Less intense than :obj:`.NORMAL`
        """


        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.Intensity:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.Intensity]:
            ...


    class AnsiFont(java.lang.Enum[VtHandler.AnsiFont]):
        """
        Modifies the shape of the font
        """

        class_: typing.ClassVar[java.lang.Class]
        NORMAL: typing.Final[VtHandler.AnsiFont]
        """
        The default font
        """

        ITALIC: typing.Final[VtHandler.AnsiFont]
        """
        Slanted or Italic font
        """

        BLACK_LETTER: typing.Final[VtHandler.AnsiFont]
        """
        Black letter or Fraktur font (hardly ever used)
        """


        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.AnsiFont:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.AnsiFont]:
            ...


    class Underline(java.lang.Enum[VtHandler.Underline]):
        """
        Places lines under the text
        """

        class_: typing.ClassVar[java.lang.Class]
        NONE: typing.Final[VtHandler.Underline]
        """
        The default, no underlines
        """

        SINGLE: typing.Final[VtHandler.Underline]
        """
        A single underline
        """

        DOUBLE: typing.Final[VtHandler.Underline]
        """
        Double underlines
        """


        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.Underline:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.Underline]:
            ...


    class Blink(java.lang.Enum[VtHandler.Blink]):
        """
        Causes text to blink
         
         
        
        If implemented, renderers should take care not to irritate the user. One option is to make
        :obj:`.FAST` actually slow, and :obj:`.SLOW` even slower. Another option is to only blink
        for a relatively short period after displaying the text, or perhaps only when the terminal
        has focus.
        """

        class_: typing.ClassVar[java.lang.Class]
        NONE: typing.Final[VtHandler.Blink]
        """
        The default, no blinking
        """

        SLOW: typing.Final[VtHandler.Blink]
        """
        Slow blinking
        """

        FAST: typing.Final[VtHandler.Blink]
        """
        Fast blinking
        """


        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.Blink:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.Blink]:
            ...


    class Direction(java.lang.Enum[VtHandler.Direction]):
        """
        A direction for relative cursor movement
        """

        class_: typing.ClassVar[java.lang.Class]
        UP: typing.Final[VtHandler.Direction]
        """
        Up a line or row
        """

        DOWN: typing.Final[VtHandler.Direction]
        """
        Down a line or row
        """

        FORWARD: typing.Final[VtHandler.Direction]
        """
        Forward or right a character or column
        """

        BACK: typing.Final[VtHandler.Direction]
        """
        Backward or left a character or column
        """


        @staticmethod
        def forCsiFinal(b: typing.Union[jpype.JByte, int]) -> VtHandler.Direction:
            """
            Derive the direction from the final byte of the CSI sequence
            
            :param jpype.JByte or int b: the final byte
            :return: the direction
            :rtype: VtHandler.Direction
            """

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.Direction:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.Direction]:
            ...


    class Erasure(java.lang.Enum[VtHandler.Erasure]):
        """
        An enumeration of erasure specifications
        """

        class_: typing.ClassVar[java.lang.Class]
        TO_LINE_END: typing.Final[VtHandler.Erasure]
        """
        Erase the current line from the cursor to the end, including the cursor's current column
        """

        TO_LINE_START: typing.Final[VtHandler.Erasure]
        """
        Erase the current line from the start to the cursor, including the cursor's current
        column
        """

        FULL_LINE: typing.Final[VtHandler.Erasure]
        """
        Erase the current line, entirely
        """

        TO_DISPLAY_END: typing.Final[VtHandler.Erasure]
        """
        Erase the current line from the cursor to the end, including the cursor's current column,
        as well as all lines after the current line.
        """

        TO_DISPLAY_START: typing.Final[VtHandler.Erasure]
        """
        Erase the current line from the start to the cursor, including the cursor's current
        column, as well as all lines before the current line. This excludes the scroll-back
        buffer.
        """

        FULL_DISPLAY: typing.Final[VtHandler.Erasure]
        """
        Erase the entire display, except the scroll-back buffer.
        """

        FULL_DISPLAY_AND_SCROLLBACK: typing.Final[VtHandler.Erasure]
        """
        Erase the entire display, including the scroll-back buffer.
        """


        @staticmethod
        def fromED(n: typing.Union[jpype.JInt, int]) -> VtHandler.Erasure:
            """
            Derive the erasure specification from the parameter to the Erase Display (ED) control
            sequence
            
            :param jpype.JInt or int n: the parameter
            :return: the erasure specification
            :rtype: VtHandler.Erasure
            """

        @staticmethod
        def fromEL(n: typing.Union[jpype.JInt, int]) -> VtHandler.Erasure:
            """
            Derive the erasure specification from the parameter to the Erase Line (EL) control
            sequence
            
            :param jpype.JInt or int n: the parameter
            :return: the erasure specification
            :rtype: VtHandler.Erasure
            """

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.Erasure:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.Erasure]:
            ...


    class KeyMode(java.lang.Enum[VtHandler.KeyMode]):
        """
        For cursor and keypad, specifies normal or application mode
         
         
        
        This affects the codes sent by the terminal.
        """

        class_: typing.ClassVar[java.lang.Class]
        NORMAL: typing.Final[VtHandler.KeyMode]
        APPLICATION: typing.Final[VtHandler.KeyMode]

        def choose(self, normal: T, application: T) -> T:
            ...

        @staticmethod
        def valueOf(name: typing.Union[java.lang.String, str]) -> VtHandler.KeyMode:
            ...

        @staticmethod
        def values() -> jpype.JArray[VtHandler.KeyMode]:
            ...


    class UnknownCsiException(java.lang.RuntimeException):
        """
        An exception for when a CSI sequence is not implemented or recognized
        """

        class_: typing.ClassVar[java.lang.Class]

        def __init__(self):
            ...


    class UnknownOscException(java.lang.RuntimeException):
        """
        An exception for when an OSC sequence is not implemented or recognized
        """

        class_: typing.ClassVar[java.lang.Class]

        def __init__(self):
            ...


    class_: typing.ClassVar[java.lang.Class]
    _4: typing.Final[jpype.JArray[jpype.JByte]]
    Q1: typing.Final[jpype.JArray[jpype.JByte]]
    Q7: typing.Final[jpype.JArray[jpype.JByte]]
    Q12: typing.Final[jpype.JArray[jpype.JByte]]
    Q25: typing.Final[jpype.JArray[jpype.JByte]]
    Q47: typing.Final[jpype.JArray[jpype.JByte]]
    Q1000: typing.Final[jpype.JArray[jpype.JByte]]
    Q1004: typing.Final[jpype.JArray[jpype.JByte]]
    Q1034: typing.Final[jpype.JArray[jpype.JByte]]
    Q1047: typing.Final[jpype.JArray[jpype.JByte]]
    Q1048: typing.Final[jpype.JArray[jpype.JByte]]
    Q1049: typing.Final[jpype.JArray[jpype.JByte]]
    Q2004: typing.Final[jpype.JArray[jpype.JByte]]
    PAT_OSC_WINDOW_TITLE: typing.Final[java.util.regex.Pattern]
    """
    Pattern for the OSC set window title sequence
    """

    PAT_OSC_COLOR_QUERY: typing.Final[java.util.regex.Pattern]
    """
    Pattern for the OSC color query sequence
    """


    @staticmethod
    def ascii(str: typing.Union[java.lang.String, str]) -> jpype.JArray[jpype.JByte]:
        """
        Use for initializing static final byte array fields from an ASCII-encoded string
        
        :param java.lang.String or str str: the string
        :return: the encoded bytes
        :rtype: jpype.JArray[jpype.JByte]
        """

    @staticmethod
    def bufEq(buf: java.nio.ByteBuffer, arr: jpype.JArray[jpype.JByte]) -> bool:
        """
        Check if the given buffer's contents are equal to that of the given array
        
        :param java.nio.ByteBuffer buf: the buffer
        :param jpype.JArray[jpype.JByte] arr: the array
        :return: true if equal, false otherwise
        :rtype: bool
        """

    @staticmethod
    def charInfo(b: typing.Union[jpype.JByte, int]) -> str:
        """
        Render a character and its byte value as a string, used for diagnostics
        
        :param jpype.JByte or int b: the byte/character to examine
        :return: the string
        :rtype: str
        """

    def decode8BitColor(self, code: typing.Union[jpype.JInt, int]) -> VtHandler.AnsiColor:
        """
        Decode the 8-bit ANSI color.
         
         
        
        Colors 0-15 are the standard + high-intensity. Colors 16-231 come from a 6x6x6 RGB cube.
        Finally, colors 232-255 are 24 steps of gray scale.
        
        :param jpype.JInt or int code: an 8-bit number
        :return: the ANSI color
        :rtype: VtHandler.AnsiColor
        """

    def decodeColor(self, colorCode: typing.Union[jpype.JInt, int], bits: java.util.PrimitiveIterator.OfInt, intensity: VtHandler.Intensity) -> VtHandler.AnsiColor:
        """
        Decode an ANSI color specification
        
        :param jpype.JInt or int colorCode: the color code (0-7 for standard, 8 for extended, 9 for default
        :param java.util.PrimitiveIterator.OfInt bits: the parameters for extended colors
        :param VtHandler.Intensity intensity: the current intensity, if applicable
        :return: the color specification
        :rtype: VtHandler.AnsiColor
        """

    def decodeExtendedColor(self, bits: java.util.PrimitiveIterator.OfInt) -> VtHandler.AnsiColor:
        """
        Decode an extended ANSI color specification
        
        :param java.util.PrimitiveIterator.OfInt bits: the parameters
        :return: the color specification
        :rtype: VtHandler.AnsiColor
        """

    def handleAltCharset(self, alt: typing.Union[jpype.JBoolean, bool]):
        """
        Handle toggling of the alternate character set.
        
        :param jpype.JBoolean or bool alt: true for G1, false for G0
        """

    def handleAltScreenBuffer(self, alt: typing.Union[jpype.JBoolean, bool], clearAlt: typing.Union[jpype.JBoolean, bool]):
        """
        Switch to and from the alternate screen buffer, optionally clearing it
         
         
        
        This will never clear the normal buffer. If the buffer does not change as a result of this
        call, then the alternate buffer is not cleared, even if clearAlt is specified.
        
        :param jpype.JBoolean or bool alt: true for alternate, false for normal
        :param jpype.JBoolean or bool clearAlt: if switching, whether to clear the alternate buffer
        """

    def handleAutoWrapMode(self, en: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle auto-wrap mode
        
        :param jpype.JBoolean or bool en: true for auto-wrap, false otherwise
        """

    def handleBackSpace(self):
        """
        Handle the backspace control code (0x08), usually just move the cursor left one.
        """

    def handleBackgroundColor(self, color: VtHandler.AnsiColor):
        """
        Handle setting of the background color
        
        :param VtHandler.AnsiColor color: the color specification
        """

    def handleBackwardTab(self, n: typing.Union[jpype.JInt, int]):
        """
        Handle the backward tab sequence: move the cursor backward n tab stops.
        
        :param jpype.JInt or int n:
        """

    def handleBell(self):
        """
        Alert the user, typically with an audible "ding" or "beep." Alternatively, a gentle visual
        alert may be used.
        """

    def handleBlink(self, blink: VtHandler.Blink):
        """
        Handle setting the blink
        
        :param VtHandler.Blink blink: the blink
        """

    def handleBlinkCursor(self, blink: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle blinking of the cursor
         
         
        
        Renderers should take care not to irritate the user. Some possibilities are to blink slowly,
        blink only for a short period of time after it moves, and/or blink only when the terminal has
        focus.
        
        :param jpype.JBoolean or bool blink: true to blink, false to leave solid
        """

    def handleBracketedPasteMode(self, en: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle bracketed paste mode
         
         
        
        See the XTerm documentation for motivation, but one example could be applications that have
        an undo stack. Without bracketed paste, the application could not recognize the pasted text
        as one undoable operation.
        
        :param jpype.JBoolean or bool en: true to bracket pasted text is special control sequences
        """

    def handleCarriageReturn(self):
        """
        Handle the carriage return control code (0x0d), usually just move the cursor to the start of
        the line.
        """

    def handleChar(self, b: typing.Union[jpype.JByte, int]):
        """
        Handle normal character output, i.e., place the character on the display
         
         
        
        This excludes control sequences and control characters, e.g., tab, line feed. While we've not
        tested, in theory, this can instead buffer the byte for decoding from UTF-8. Still, the
        implementation should eagerly decode, rendering characters as soon as they are available.
        
        :param jpype.JByte or int b: the byte/character
        :raises java.lang.Exception: if anything goes wrong
        """

    def handleCharExc(self, b: typing.Union[jpype.JByte, int]):
        """
        Handle a character not part of an escape sequence.
         
         
        
        This may include control characters, which are displatched appropriately by this method.
        Additionally, this handles any exception thrown by :meth:`handleChar(byte) <.handleChar>`.
        
        :param jpype.JByte or int b: the byte/character
        """

    def handleCsi(self, csiParam: java.nio.ByteBuffer, csiInter: java.nio.ByteBuffer, csiFinal: typing.Union[jpype.JByte, int]):
        """
        Handle a CSI sequence
        
        :param java.nio.ByteBuffer csiParam: the parameter buffer
        :param java.nio.ByteBuffer csiInter: the intermediate buffer
        :param jpype.JByte or int csiFinal: the final byte
        :raises java.lang.Exception: if anything goes wrong
        """

    def handleCsiExc(self, csiParam: java.nio.ByteBuffer, csiInter: java.nio.ByteBuffer, csiFinal: typing.Union[jpype.JByte, int]):
        """
        Handle a CSI sequence, printing any exception
        
        
        .. seealso::
        
            | :obj:`.handleCsi(ByteBuffer, ByteBuffer, byte)`
        """

    def handleCursorKeyMode(self, mode: VtHandler.KeyMode):
        """
        Toggle cursor key mode
        
        :param VtHandler.KeyMode mode: the key mode
        """

    def handleDeleteCharacters(self, n: typing.Union[jpype.JInt, int]):
        """
        Delete n characters from the current cursor position, and shift the remaining characters
        back. If n is one, only the character at the cursor position is deleted. If n is greater,
        then additional characters are deleted after (to the right) of the cursor. Consider the
        current line contents and cursor position:
         
         
        123456789
            ^
         
         
        Deleting 2 characters should result in ``1234789``. The character at the cursor (5) and
        the following character (6) are deleted. The remaining (789) are all shifted back (left).
        
        :param jpype.JInt or int n: the number of characters to delete.
        """

    def handleDeleteLines(self, n: typing.Union[jpype.JInt, int]):
        """
        Delete n lines at and below the cursor
         
         
        
        Lines within the viewport are shifted up, and new lines inserted at the bottom.
        
        :param jpype.JInt or int n: the number of lines to delete
        """

    def handleErase(self, erasure: VtHandler.Erasure):
        """
        Handle a request to erase part of the display
        
        :param VtHandler.Erasure erasure: what, relative to the cursor, to erase
        """

    def handleEraseCharacters(self, n: typing.Union[jpype.JInt, int]):
        """
        Erase n characters from the current cursor position. In essence, replace the erased
        characters with spaces. If n is one, only the character at the cursor position is erased. If
        n is greater, then additional characters are erased after (to the right) of the cursor.
        
        :param jpype.JInt or int n: the number of characters to erase.
        """

    def handleFont(self, font: VtHandler.AnsiFont):
        """
        Handle setting the font
        
        :param VtHandler.AnsiFont font: the font
        """

    def handleForegroundColor(self, color: VtHandler.AnsiColor):
        """
        Handle setting of the foreground color
        
        :param VtHandler.AnsiColor color: the color specification
        """

    def handleFullReset(self):
        """
        Handle a request to fully reset the terminal
         
         
        
        All buffers should be cleared and all state variables, positions, attributes, etc., should be
        reset to their defaults.
        """

    def handleHOrLStuff(self, csiParam: java.nio.ByteBuffer, en: typing.Union[jpype.JBoolean, bool]):
        """
        Handle the parameters for a 'h' or 'l' final byte CSI sequence
        
        :param java.nio.ByteBuffer csiParam: the parameter buffer
        :param jpype.JBoolean or bool en: true for 'h', which generally enables things, and false for 'l'
        """

    def handleHidden(self, hidden: typing.Union[jpype.JBoolean, bool]):
        """
        Handle toggling of the hidden attribute
        
        :param jpype.JBoolean or bool hidden: true to hide, false to show
        """

    def handleInsertCharacters(self, n: typing.Union[jpype.JInt, int]):
        """
        Insert n blank characters at the current cursor position, shifting characters right to make
        room.
        
        :param jpype.JInt or int n: the number of characters to insert.
        """

    def handleInsertLines(self, n: typing.Union[jpype.JInt, int]):
        """
        Insert n lines at and below the cursor
         
         
        
        Lines within the viewport are shifted down or deleted to make room for the new lines.
        
        :param jpype.JInt or int n: the number of lines to insert
        """

    def handleInsertMode(self, en: typing.Union[jpype.JBoolean, bool]):
        """
        Handle toggling insert mode
         
         
        
        In insert mode, characters at and to the right of the cursor are shifted right to make room
        for each new character. In replace mode (default), the character at the cursor is replaced
        with each new character.
        
        :param jpype.JBoolean or bool en: true for insert, false for replace (default)
        """

    def handleIntensity(self, intensity: VtHandler.Intensity):
        """
        Handle setting the intensity
        
        :param VtHandler.Intensity intensity: the intensity
        """

    def handleKeypadMode(self, mode: VtHandler.KeyMode):
        """
        Toggle keypad mode
        
        :param VtHandler.KeyMode mode: the key mode
        """

    def handleLineFeed(self):
        """
        Handle the line feed control code (0x0a), usually just move the cursor down one.
        """

    def handleMetaKey(self, en: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle handling of the meta key
        
        :param jpype.JBoolean or bool en: true to report the meta modifier in key/mouse events, false to exclude it
        """

    @typing.overload
    def handleMoveCursor(self, direction: VtHandler.Direction, n: typing.Union[jpype.JInt, int]):
        """
        Handle a relative cursor movement command
        
        :param VtHandler.Direction direction: the direction
        :param jpype.JInt or int n: the number of rows or columns to move
        """

    @typing.overload
    def handleMoveCursor(self, row: typing.Union[jpype.JInt, int], col: typing.Union[jpype.JInt, int]):
        """
        Handle an absolute cursor movement command
        
        :param jpype.JInt or int row: the row (0-up)
        :param jpype.JInt or int col: the column (0-up)
        """

    def handleMoveCursorCol(self, col: typing.Union[jpype.JInt, int]):
        """
        Handle an absolute cursor column movement command
        
        :param jpype.JInt or int col: the column (0-up)
        """

    def handleMoveCursorRow(self, row: typing.Union[jpype.JInt, int]):
        """
        Handle an absolute cursor row movement command
         
         
        
        The column should remain the same, i.e., do *not* reset the column to 0.
        
        :param jpype.JInt or int row: the row (0-up)
        """

    def handleOsc(self, oscParam: java.nio.ByteBuffer):
        """
        Handle an OSC sequence
        
        :param java.nio.ByteBuffer oscParam: the parameter buffer
        :raises java.lang.Exception: if anything goes wrong
        """

    def handleOscExc(self, oscParam: java.nio.ByteBuffer):
        """
        Handle a OSC sequence, printing any exception
        
        
        .. seealso::
        
            | :obj:`.handleOsc(ByteBuffer)`
        """

    def handleProportionalSpacing(self, spacing: typing.Union[jpype.JBoolean, bool]):
        """
        Handle setting proportional spacing
        
        :param jpype.JBoolean or bool spacing: true for space proportionally, false otherwise
        """

    def handleReportCursorPos(self):
        """
        Handle a request to report the cursor position
        """

    def handleReportFocus(self, report: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle reporting of terminal focus
        
        :param jpype.JBoolean or bool report: true to report focus gain and loss events, false to not report them
        """

    def handleReportMouseEvents(self, press: typing.Union[jpype.JBoolean, bool], release: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle reporting of select mouse events
        
        :param jpype.JBoolean or bool press: true to report mouse press events, false to not report them
        :param jpype.JBoolean or bool release: true to report mouse release events, false to not report them
        """

    def handleResetAttributes(self):
        """
        Handle resetting the SGR attributes
        """

    def handleRestoreCursorPos(self):
        """
        Handle a request to restore the previously-saved cursor position
        """

    def handleRestoreIconTitle(self):
        """
        Handle a request to restore the terminal window's icon title
         
         
        
        The title is set to the one popped from the stack of saved window icon titles.
        
        
        .. seealso::
        
            | :obj:`.handleSaveIconTitle()`
        """

    def handleRestoreWindowTitle(self):
        """
        Handle a request to restore the terminal window's title
         
         
        
        The title is set to the one popped from the stack of saved window titles.
        
        
        .. seealso::
        
            | :obj:`.handleSaveWindowTitle()`
        """

    def handleReverseVideo(self, reverse: AnsiColorResolver.ReverseVideo):
        """
        Handle toggling of reverse video
         
         
        
        This can be a bit confusing with default colors. In general, this means swapping the
        foreground and background color specifications (not inverting the colors or mirroring or some
        such). In the case of the default colors, the implementor must be sure to swap the meaning or
        "default background" and "default foreground." Furthermore, if "do not paint" is used for
        "default background," care must be taken to ensure the foreground is still painted in
        reversed mode.
        
        :param AnsiColorResolver.ReverseVideo reverse: the reverse video mode
        """

    def handleSaveCursorPos(self):
        """
        Handle a request to save the cursor position
        """

    def handleSaveIconTitle(self):
        """
        Handle a request to save the terminal window's icon title
         
         
        
        "Icon titles" are a concept from the X Windows system. Do the closest equivalent, if anything
        applies at all. The current title is pushed to a stack of limited size.
        """

    def handleSaveWindowTitle(self):
        """
        Handle a request to save the terminal window's title
         
         
        
        Window titles are fairly applicable to all desktop windowing systems. The current title is
        pushed to a stack of limited size.
        """

    def handleScrollLinesDown(self, n: typing.Union[jpype.JInt, int]):
        """
        Scroll the lines n slots down, considering only those lines in the scrolling range.
         
         
        
        This is equivalent to scrolling the *viewport* n lines *up*. This method exists
        in attempt to reflect "up" and "down" correctly in the documentation. Unfortunately, the
        documentation is not always clear whether we're scrolling the viewport or the lines
        themselves.
        
        :param jpype.JInt or int n: the number of lines to scroll
        
        .. seealso::
        
            | :obj:`.handleScrollViewportUp(int)`
        """

    def handleScrollLinesUp(self, n: typing.Union[jpype.JInt, int], intoScrollBack: typing.Union[jpype.JBoolean, bool]):
        """
        Scroll the lines n slots up, considering only those lines in the scrolling range.
         
         
        
        The is equivalent to scrolling the *viewport* n lines *down*. This method
        exists in attempt to reflect "up" and "down" correctly in the documentation. Unfortunately,
        the documentation is not always clear whether we're scrolling the viewport or the lines
        themselves.
        
        :param jpype.JInt or int n: the number of lines to scroll
        :param jpype.JBoolean or bool intoScrollBack: specifies whether the top line may flow into the scroll-back buffer
        
        .. seealso::
        
            | :obj:`.handleScrollViewportDown(int)`
        """

    def handleScrollViewportDown(self, n: typing.Union[jpype.JInt, int], intoScrollBack: typing.Union[jpype.JBoolean, bool]):
        """
        Scroll the display n lines down, considering only those lines in the scrolling range.
         
         
        
        To be unambiguous, this of movement of the viewport. The viewport scrolls down, so the lines
        themselves scroll up. The default range is the whole display. The cursor is not moved.
        
        :param jpype.JInt or int n: the number of lines to scroll
        :param jpype.JBoolean or bool intoScrollBack: specifies whether the top line may flow into the scroll-back buffer
        
        .. seealso::
        
            | :obj:`.handleSetScrollRange(Integer, Integer)`
        """

    def handleScrollViewportUp(self, n: typing.Union[jpype.JInt, int]):
        """
        Scroll the display n lines up, considering only those lines in the scrolling range.
        
        :param jpype.JInt or int n: the number of lines to scroll
        
        .. seealso::
        
            | :obj:`.handleScrollDown()`
        
            | :obj:`.handleSetScrollRange(Integer, Integer)`
        """

    def handleSetCharset(self, g: VtCharset.G, cs: VtCharset):
        """
        Set the charset for a given slot
        
        :param VtCharset.G g: the slot
        :param VtCharset cs: the charset
        """

    def handleSetScrollRange(self, start: typing.Union[java.lang.Integer, int], end: typing.Union[java.lang.Integer, int]):
        """
        Set the range of rows (viewport) involved in scrolling.
         
         
        
        This applies not only to :meth:`handleScrollUp() <.handleScrollUp>` and :meth:`handleScrollDown() <.handleScrollDown>`, but also
        to when the cursor moves far enough down that the display must scroll. Normally, start is 0
        and end is rows-1 (The parser will adjust the 1-up indices to 0-up) so that the entire
        display is scrolled. If the cursor moves past end (not just the end of the device, but the
        end given here) then the scrolling region must be scrolled. The top line is removed, the
        interior lines are moved up, and the bottom line is cleared. If the terminal is resized, the
        scroll range is reset to the whole display.
        
        :param java.lang.Integer or int start: the first row (0-up) in the scrolling region. If omitted, the first row of the
                    display.
        :param java.lang.Integer or int end: the last row (0-up, inclusive) in the scrolling region. If omitted, the last row
                    of the display.
        """

    def handleSgrAttribute(self, bits: java.util.PrimitiveIterator.OfInt):
        """
        Handle an Select Graphics Rendition attribute (final byte 'm')
        
        :param java.util.PrimitiveIterator.OfInt bits: the parameters
        """

    def handleShowCursor(self, show: typing.Union[jpype.JBoolean, bool]):
        """
        Toggle display of the cursor
        
        :param jpype.JBoolean or bool show: true to show the cursor, false to hide it.
        """

    def handleStrikeThrough(self, strikeThrough: typing.Union[jpype.JBoolean, bool]):
        """
        Handle setting strike-through
        
        :param jpype.JBoolean or bool strikeThrough: true to strike, false for no strike
        """

    def handleTab(self):
        """
        Handle the tab control code (0x09), usually just move the cursor to the next tab stop.
        """

    def handleUnderline(self, underline: VtHandler.Underline):
        """
        Handle setting the underline
        
        :param VtHandler.Underline underline: the underline
        """

    def handleWindowManipulation(self, csiParam: java.nio.ByteBuffer):
        """
        Handle XTerm CSI commands that manipulate the window titles
        
        :param java.nio.ByteBuffer csiParam: the buffer of parameters
        """

    def handleWindowTitle(self, title: typing.Union[java.lang.String, str]):
        """
        Handle a request to set the terminal window's title
        
        :param java.lang.String or str title: the titled
        """

    @staticmethod
    def parseCsiInts(csiParam: java.nio.ByteBuffer) -> java.util.PrimitiveIterator.OfInt:
        """
        Parse a sequence of integers in the form ``<em>n</em> ; <em>m</em> ;`` ....
         
         
        
        This is designed to replace the :meth:`String.split(String) <String.split>` and
        :meth:`Integer.parseInt(String) <Integer.parseInt>` pattern, which should avoid some unnecessary object
        creation. Unfortunately, the iterator itself is still an object.... Each parameter is parsed
        on demand.
        
        :param java.nio.ByteBuffer csiParam: the buffer of characters containing the parameters to parse
        :return: an iterator of integers
        :rtype: java.util.PrimitiveIterator.OfInt
        """

    @staticmethod
    def strBuf(buf: java.nio.ByteBuffer) -> str:
        """
        Decode the byte buffer's contents to an ASCII string, used for diagnostics.
        
        :param java.nio.ByteBuffer buf: the buffer to examine
        :return: the string
        :rtype: str
        """

    @staticmethod
    def truncateAtNull(str: typing.Union[java.lang.String, str]) -> str:
        ...


class VtLine(java.lang.Object):
    """
    A line of text in the :obj:`VtBuffer`
    """

    class RunConsumer(java.lang.Object):
        """
        A callback for a run of contiguous characters having the same attributes
        """

        class_: typing.ClassVar[java.lang.Class]

        def accept(self, attrs: VtAttributes, start: typing.Union[jpype.JInt, int], end: typing.Union[jpype.JInt, int]):
            """
            Execute an action on a run
            
            :param VtAttributes attrs: the attributes shared by all in the run
            :param jpype.JInt or int start: the first column of the run, 0 up
            :param jpype.JInt or int end: the last column of the run, exclusive, 0 up
            """


    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, cols: typing.Union[jpype.JInt, int]):
        """
        Create a line with the given maximum number of characters
        
        :param jpype.JInt or int cols: the maximum number of characters
        """

    def clear(self):
        """
        Clear the full line
        """

    def clearToEnd(self, x: typing.Union[jpype.JInt, int]):
        """
        Clear characters at and after the given column
        
        :param jpype.JInt or int x: the column, 0 up
        """

    def clearToStart(self, x: typing.Union[jpype.JInt, int], attrs: VtAttributes):
        """
        Clear characters before and at the given column
        
        :param jpype.JInt or int x: the column, 0 up
        :param VtAttributes attrs: attributes to apply to the cleared (space) characters
        """

    def cols(self) -> int:
        """
        Get the number of columns in the line
        
        :return: the column count
        :rtype: int
        """

    def delete(self, start: typing.Union[jpype.JInt, int], end: typing.Union[jpype.JInt, int]):
        """
        Delete characters in the given range, shifting remaining characters to the left
        
        :param jpype.JInt or int start: the first column, 0 up
        :param jpype.JInt or int end: the last column, exclusive, 0 up
        """

    def erase(self, start: typing.Union[jpype.JInt, int], end: typing.Union[jpype.JInt, int], attrs: VtAttributes):
        """
        Replace characters in the given range with spaces
         
         
        
        If the last column is erased, this instead clears from the start to the end. The difference
        is subtle, but deals in how the line reports its text contents. The trailing spaces will not
        be included if this call results in the last column being erased.
        
        :param jpype.JInt or int start: the first column, 0 up
        :param jpype.JInt or int end: the last column, exclusive, 0 up
        :param VtAttributes attrs: the attributes to assign the space characters
        """

    def findWord(self, x: typing.Union[jpype.JInt, int], forward: typing.Union[jpype.JBoolean, bool]) -> int:
        """
        Find the boundaries for the word at the given column
        
        :param jpype.JInt or int x: the column, 0 up
        :param jpype.JBoolean or bool forward: true to find the end, false to find the beginning
        :return: the first column, 0 up, or the last column, exclusive, 0 up
        :rtype: int
        """

    def forEachRun(self, action: VtLine.RunConsumer):
        """
        Execute an action on each run of contiguous characters having the same attributes, from left
        to right.
        
        :param VtLine.RunConsumer action: the callback action
        """

    def gatherText(self, sb: java.lang.StringBuilder, start: typing.Union[jpype.JInt, int], end: typing.Union[jpype.JInt, int]):
        """
        Append a portion of this line's text to the given string builder
        
        :param java.lang.StringBuilder sb: the destination builder
        :param jpype.JInt or int start: the first column, 0 up
        :param jpype.JInt or int end: the last column, exclusive, 0 up
        """

    def getCellAttrs(self, x: typing.Union[jpype.JInt, int]) -> VtAttributes:
        """
        Get the attributes for the character in the given column
        
        :param jpype.JInt or int x: the column, 0 up
        :return: the attributes
        :rtype: VtAttributes
        """

    def getChar(self, x: typing.Union[jpype.JInt, int]) -> str:
        """
        Get the character in the given column
        
        :param jpype.JInt or int x: the column, 0 up
        :return: the character
        :rtype: str
        """

    def getCharBuffer(self) -> jpype.JArray[jpype.JChar]:
        """
        Get the full character buffer
         
         
        
        This is a reference to the buffer, which is very useful when rendering. Modifying this buffer
        externally is not recommended.
        
        :return: the buffer
        :rtype: jpype.JArray[jpype.JChar]
        """

    def insert(self, start: typing.Union[jpype.JInt, int], n: typing.Union[jpype.JInt, int]):
        """
        Insert n (space) characters at and after the given column
        
        :param jpype.JInt or int start: the column, 0 up
        :param jpype.JInt or int n: the number of characters to insert
        """

    @staticmethod
    def isWordChar(ch: typing.Union[jpype.JChar, int, str]) -> bool:
        """
        Check if the given character is considered part of a word
         
         
        
        This is used both when selecting words, and when requiring search to find whole words.
        
        :param jpype.JChar or int or str ch: the character
        :return: true if the character is part of a word
        :rtype: bool
        """

    def length(self) -> int:
        """
        Get the length of the line, excluding trailing cleared characters
        
        :return: the length
        :rtype: int
        """

    def putChar(self, x: typing.Union[jpype.JInt, int], c: typing.Union[jpype.JChar, int, str], attrs: VtAttributes):
        """
        Place the given character with attributes into the given column
        
        :param jpype.JInt or int x: the column, 0 up
        :param jpype.JChar or int or str c: the character
        :param VtAttributes attrs: the attributes
        """

    def reset(self, cols: typing.Union[jpype.JInt, int]):
        """
        Reset the line
        
        :param jpype.JInt or int cols:
        """

    def resize(self, cols: typing.Union[jpype.JInt, int]):
        """
        Resize the line to the given maximum character count
        
        :param jpype.JInt or int cols: the maximum number of characters
        """

    @property
    def cellAttrs(self) -> VtAttributes:
        ...

    @property
    def char(self) -> jpype.JChar:
        ...

    @property
    def charBuffer(self) -> jpype.JArray[jpype.JChar]:
        ...


class VtOutput(java.lang.Object):
    """
    A callback for bytes generated by the terminal, e.g., to report a key press or reply to a
    request.
    """

    class_: typing.ClassVar[java.lang.Class]

    def out(self, buf: java.nio.ByteBuffer):
        """
        Handle output from the terminal
         
         
        
        Most likely these bytes should be sent down an output stream, usually to a pty.
        
        :param java.nio.ByteBuffer buf: the buffer of bytes generated
        """


class VtParser(java.lang.Object):
    """
    The parser for a terminal emulator
     
     
    
    The only real concern of this parser is to separate escape sequences from normal character
    output. All state not related to parsing is handled by a :obj:`VtHandler`. Most of the logic is
    implemented in the machine state nodes: :obj:`VtState`.
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, handler: VtHandler):
        """
        Construct a parser with the given handler
        
        :param VtHandler handler: the handler
        """

    def process(self, buf: java.nio.ByteBuffer):
        """
        Process the bytes from the given buffer
         
         
        
        This is likely fed from an input stream, usually of a pty.
        
        :param java.nio.ByteBuffer buf: the buffer
        """


class VtResponseEncoder(java.lang.Object):

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, charset: java.nio.charset.Charset):
        ...

    def reportCursorPos(self, row: typing.Union[jpype.JInt, int], col: typing.Union[jpype.JInt, int]):
        ...

    def reportPasteEnd(self):
        ...

    def reportPasteStart(self):
        ...


class VtState(java.lang.Enum[VtState]):

    class_: typing.ClassVar[java.lang.Class]
    CHAR: typing.Final[VtState]
    """
    The initial state, just process output characters until we encounter an ``ESC``.
    """

    ESC: typing.Final[VtState]
    """
    We have just encountered an ``ESC``.
    """

    CHARSET: typing.Final[VtState]
    """
    We have encountered ``ESC`` and a charset-selection byte. Now we just need to know the
    charset. Most are one byte, but there are some two-byte codes.
    """

    CHARSET_QUOTE: typing.Final[VtState]
    """
    We're selecting a two-byte charset, and we just encountered ``"``.
    """

    CHARSET_PERCENT: typing.Final[VtState]
    """
    We're selecting a two-byte charset, and we just encountered ``%``.
    """

    CHARSET_AMPERSAND: typing.Final[VtState]
    """
    We're selecting a two-byte charset, and we just encountered ``&``.
    """

    CSI_PARAM: typing.Final[VtState]
    """
    We've encountered ``CSI``, so now we're parsing parameters, intermediates, or the final
    character.
    """

    CSI_INTER: typing.Final[VtState]
    """
    We've finished (or skipped) parsing CSI parameters, so now we're parsing intermediates or the
    final character.
    """

    OSC_PARAM: typing.Final[VtState]
    """
    We've encountered ``OSC``, so now we're parsing parameters until we encounter ``BEL``
    or ``ST``.
    """

    OSC_ESC: typing.Final[VtState]
    """
    We've encountered ``ESC`` part of , so now we're parsing parameters until we encounter
    ``BEL`` or ``ST``.
    """


    @staticmethod
    def valueOf(name: typing.Union[java.lang.String, str]) -> VtState:
        ...

    @staticmethod
    def values() -> jpype.JArray[VtState]:
        ...



__all__ = ["AnsiColorResolver", "VtAttributes", "VtBuffer", "VtCharset", "VtHandler", "VtLine", "VtOutput", "VtParser", "VtResponseEncoder", "VtState"]
