/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin.format.elf.info;

import ghidra.app.cmd.comments.AppendCommentCmd;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteArrayProvider;
import ghidra.app.util.bin.StructConverter;
import ghidra.app.util.bin.format.elf.info.ElfInfoItem;
import ghidra.app.util.bin.format.elf.info.StandardElfInfoProducer;
import ghidra.program.model.address.Address;
import ghidra.program.model.data.ArrayDataType;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeComponent;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.DataUtilities;
import ghidra.program.model.data.LEB128;
import ghidra.program.model.data.PointerDataType;
import ghidra.program.model.data.Structure;
import ghidra.program.model.data.StructureDataType;
import ghidra.program.model.listing.CodeUnit;
import ghidra.program.model.listing.CommentType;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Listing;
import ghidra.program.model.listing.Program;
import ghidra.program.model.symbol.RefType;
import ghidra.program.model.symbol.ReferenceManager;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.SymbolUtilities;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.util.Msg;
import ghidra.util.exception.InvalidInputException;
import java.io.IOException;
import java.util.Objects;

public class GnuBuildAttribute
implements ElfInfoItem {
    private static final String GNU_ATTR_NAME_PREFIX = "GA";
    private static final int NAME_PREFIX_LEN;
    private static final int KIND_OFFSET;
    private static final int ID_OFFSET;
    private final int nameLen;
    private final byte[] nameBytes;
    private final int vendorType;
    private final byte[] description;

    public static GnuBuildAttribute read(BinaryReader reader) throws IOException {
        int nameLen = reader.readNextUnsignedIntExact();
        int descLen = reader.readNextUnsignedIntExact();
        int vendorType = reader.readNextInt();
        if (nameLen > 1024 || descLen > 0x100000) {
            throw new IOException("Invalid Note lengths: %d, %d".formatted(nameLen, descLen));
        }
        byte[] nameBytes = reader.readNextByteArray(nameLen);
        byte[] desc = reader.readNextByteArray(descLen);
        return new GnuBuildAttribute(nameLen += reader.align(4), nameBytes, vendorType, desc);
    }

    public GnuBuildAttribute(int nameLen, byte[] nameBytes, int vendorType, byte[] description) {
        this.nameLen = nameLen;
        this.nameBytes = nameBytes;
        this.vendorType = vendorType;
        this.description = description;
    }

    public AttributeId getId() {
        if (this.nameBytes.length < ID_OFFSET + 1) {
            return null;
        }
        int ch = Byte.toUnsignedInt(this.nameBytes[ID_OFFSET]);
        return AttributeId.of(ch);
    }

    public String getIdString() {
        AttributeId id = this.getId();
        if (id == AttributeId.STRINGVAL) {
            try {
                String s = this.getNameReader(true).readUtf8String(ID_OFFSET);
                return "\"%s\"".formatted(s);
            }
            catch (IOException e) {
                return "unknown";
            }
        }
        return id != null ? id.name() : "unknown";
    }

    public Object getValue(Program program) {
        try {
            ValueType valtype = this.getValueType();
            if (valtype == null) {
                return null;
            }
            BinaryReader reader = this.getNameReader(!program.getMemory().isBigEndian());
            reader.setPointerIndex(this.getValueOffset());
            return switch (valtype.ordinal()) {
                case 1 -> reader.readNextUtf8String();
                case 0 -> reader.readNext(LEB128::unsigned);
                case 2 -> false;
                case 3 -> true;
                default -> null;
            };
        }
        catch (IOException e) {
            return null;
        }
    }

    private int getValueOffset() {
        AttributeId id = this.getId();
        if (id == AttributeId.STRINGVAL) {
            BinaryReader reader = this.getNameReader(true);
            try {
                reader.setPointerIndex(ID_OFFSET);
                reader.readNextUtf8String();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return (int)reader.getPointerIndex();
        }
        return ID_OFFSET + 1;
    }

    private ValueType getValueType() {
        return this.nameBytes.length > NAME_PREFIX_LEN ? ValueType.of(Byte.toUnsignedInt(this.nameBytes[NAME_PREFIX_LEN])) : null;
    }

    public AttributeType getAttributeType() {
        return AttributeType.of(this.vendorType);
    }

    public AddressPair getRange(Program program) {
        int ptrSize = program.getDefaultPointerSize();
        if (this.description.length == ptrSize * 2) {
            try {
                BinaryReader reader = this.getDescriptionReader(!program.getMemory().isBigEndian());
                long startOfs = reader.readNextUnsignedValue(ptrSize);
                long endOfs = reader.readNextUnsignedValue(ptrSize);
                Address startAddr = program.getImageBase().getNewAddress(startOfs);
                Address endAddr = program.getImageBase().getNewAddress(endOfs - 1L);
                return new AddressPair(startAddr, endAddr);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return null;
    }

    public String getLabel(Program program) {
        String idStr = this.getIdString();
        Object val = this.getValue(program);
        String valStr = val != null ? val.toString() : "unknown";
        AttributeType type = this.getAttributeType();
        String typeStr = type != null ? type.name() : "unknown";
        return "gnu.build.attribute_%s_%s=%s".formatted(typeStr, idStr, valStr);
    }

    public String getDescription(Program program) {
        String idStr = this.getIdString();
        Object val = this.getValue(program);
        String valStr = val != null ? val.toString() : "unknown";
        return "%s=%s".formatted(idStr, valStr);
    }

    @Override
    public void markupProgram(Program program, Address address) {
        StructureDataType dt = this.createNoteStructure(StandardElfInfoProducer.ELF_CATEGORYPATH, "GnuBuildAttribute", program, (DataTypeManager)program.getDataTypeManager());
        if (dt != null) {
            try {
                AddressPair range = this.getRange(program);
                Object attrComment = this.getDescription(program);
                if (range != null) {
                    attrComment = (String)attrComment + ", range=" + range.toString();
                }
                GnuBuildAttribute.appendComment(program, address, CommentType.EOL, "", (String)attrComment, "\n");
                String label = SymbolUtilities.replaceInvalidChars((String)this.getLabel(program), (boolean)true);
                program.getSymbolTable().createLabel(address, label, null, SourceType.IMPORTED);
                DataUtilities.createData((Program)program, (Address)address, (DataType)dt, (int)-1, (boolean)false, (DataUtilities.ClearDataMode)DataUtilities.ClearDataMode.CLEAR_ALL_UNDEFINED_CONFLICT_DATA);
                if (range != null) {
                    Address startFieldAddr = this.getFieldAddr((Structure)dt, "start", address);
                    Address endFieldAddr = this.getFieldAddr((Structure)dt, "end", address);
                    if (startFieldAddr != null && endFieldAddr != null) {
                        ReferenceManager refMgr = program.getReferenceManager();
                        refMgr.addMemoryReference(startFieldAddr, range.first(), RefType.DATA, SourceType.IMPORTED, 0);
                        refMgr.addMemoryReference(endFieldAddr, range.second(), RefType.DATA, SourceType.IMPORTED, 0);
                    }
                }
            }
            catch (CodeUnitInsertionException | InvalidInputException e) {
                Msg.error((Object)this, (Object)"Failed to markup Elf Note at %s: %s".formatted(address, this), (Throwable)e);
            }
        }
    }

    private Address getFieldAddr(Structure struct, String fieldName, Address startAddr) {
        for (DataTypeComponent dtc : struct.getDefinedComponents()) {
            if (!fieldName.equals(dtc.getFieldName())) continue;
            return startAddr.add((long)dtc.getOffset());
        }
        return null;
    }

    private BinaryReader getNameReader(boolean isLittleEndian) {
        ByteArrayProvider bap = new ByteArrayProvider(this.nameBytes);
        BinaryReader descReader = new BinaryReader(bap, isLittleEndian);
        return descReader;
    }

    private BinaryReader getDescriptionReader(boolean isLittleEndian) {
        ByteArrayProvider bap = new ByteArrayProvider(this.description);
        BinaryReader descReader = new BinaryReader(bap, isLittleEndian);
        return descReader;
    }

    private StructureDataType createNoteStructure(CategoryPath cp, String structName, Program program, DataTypeManager dtm) {
        cp = Objects.requireNonNullElse(cp, StandardElfInfoProducer.ELF_CATEGORYPATH);
        StructureDataType result = new StructureDataType(cp, "%s_%d_%d".formatted(structName, this.nameBytes.length, this.description.length), 0, dtm);
        result.add(StructConverter.DWORD, "namesz", "Length of name field");
        result.add(StructConverter.DWORD, "descsz", "Length of description field");
        result.add(StructConverter.DWORD, "type", "Vendor specific type");
        result.add((DataType)new ArrayDataType(StructConverter.ASCII, this.nameLen), "name", null);
        int ptrSize = program.getDefaultPointerSize();
        if (this.description.length == ptrSize * 2) {
            result.add((DataType)new PointerDataType(), "start", null);
            result.add((DataType)new PointerDataType(), "end", null);
        } else if (this.description.length != 0) {
            result.add((DataType)new ArrayDataType(StructConverter.BYTE, this.description.length), "unknown", null);
        }
        return result;
    }

    static void appendComment(Program program, Address address, CommentType commentType, String prefix, String comment, String sep) {
        String existingComment;
        if (comment == null || comment.isBlank()) {
            return;
        }
        CodeUnit cu = GnuBuildAttribute.getCodeUnitForComment(program, address);
        if (cu != null && (existingComment = cu.getComment(commentType)) != null && existingComment.contains(comment)) {
            return;
        }
        AppendCommentCmd cmd = new AppendCommentCmd(address, commentType, Objects.requireNonNullElse(prefix, "") + comment, sep);
        cmd.applyTo(program);
    }

    static CodeUnit getCodeUnitForComment(Program program, Address address) {
        Listing listing = program.getListing();
        CodeUnit cu = listing.getCodeUnitContaining(address);
        if (cu == null) {
            return null;
        }
        Address cuAddr = cu.getMinAddress();
        if (cu instanceof Data && !address.equals((Object)cuAddr)) {
            Data data = (Data)cu;
            return data.getPrimitiveAt((int)address.subtract(cuAddr));
        }
        return cu;
    }

    static {
        KIND_OFFSET = NAME_PREFIX_LEN = GNU_ATTR_NAME_PREFIX.length();
        ID_OFFSET = KIND_OFFSET + 1;
    }

    static enum AttributeId {
        RESERVED(0),
        VERSION(1),
        STACK_PROT(2),
        RELRO(3),
        STACKSIZE(4),
        TOOL(5),
        ABI(6),
        POSITION_INDEPENDENCE(7),
        SHORT_ENUM(8),
        RESERVED1(9),
        STRINGVAL(31),
        RESERVED2(127);

        private static final int RESERVED1_START = 9;
        private static final int RESERVED1_END = 31;
        private static final int RESERVED2_START = 127;
        private static final int FIRSTCHAR_START = 32;
        private final int idVal;
        private static final AttributeId[] lookupValues;

        private AttributeId(int idVal) {
            this.idVal = idVal;
        }

        public int getIdVal() {
            return this.idVal;
        }

        public static AttributeId of(int idVal) {
            if (9 <= idVal && idVal <= 31) {
                return RESERVED1;
            }
            if (32 <= idVal && idVal < 127) {
                return STRINGVAL;
            }
            if (127 <= idVal) {
                return RESERVED2;
            }
            for (AttributeId e : lookupValues) {
                if (e.idVal != idVal) continue;
                return e;
            }
            return null;
        }

        static {
            lookupValues = AttributeId.values();
        }
    }

    static enum ValueType {
        NUMERIC('*', "num"),
        STRING('$', "str"),
        BOOLEAN_FALSE('!', "bool"),
        BOOLEAN_TRUE('+', "bool");

        private final char ch;
        private final String abbr;
        private static final ValueType[] lookupValues;

        private ValueType(char ch, String abbr) {
            this.ch = ch;
            this.abbr = abbr;
        }

        public String getAbbr() {
            return this.abbr;
        }

        public static ValueType of(int ch) {
            for (ValueType e : lookupValues) {
                if (e.ch != ch) continue;
                return e;
            }
            return null;
        }

        static {
            lookupValues = ValueType.values();
        }
    }

    static enum AttributeType {
        OPEN(256, "open range"),
        FUNC(257, "func symbol range");

        private final int typeInt;
        private final String desc;
        private static final AttributeType[] lookupValues;

        private AttributeType(int typeInt, String desc) {
            this.typeInt = typeInt;
            this.desc = desc;
        }

        public int getTypeInt() {
            return this.typeInt;
        }

        public String getDescription() {
            return this.desc;
        }

        public static AttributeType of(int typeInt) {
            for (AttributeType e : lookupValues) {
                if (e.typeInt != typeInt) continue;
                return e;
            }
            return null;
        }

        static {
            lookupValues = AttributeType.values();
        }
    }

    public record AddressPair(Address first, Address second) {
        @Override
        public String toString() {
            return this.first.toString() + "-" + this.second.toString();
        }
    }
}

