/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.store.local;

import generic.timer.GhidraSwinglessTimer;
import ghidra.framework.store.FileSystem;
import ghidra.framework.store.FolderNotEmptyException;
import ghidra.framework.store.local.IndexedPropertyFile;
import ghidra.framework.store.local.ItemPropertyFile;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.framework.store.local.LocalFolderItem;
import ghidra.util.InvalidNameException;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.ReadOnlyException;
import ghidra.util.StringUtilities;
import ghidra.util.exception.DuplicateFileException;
import ghidra.util.exception.NotFoundException;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

public class IndexedLocalFileSystem
extends LocalFileSystem {
    public static final int LATEST_INDEX_VERSION = 1;
    static final int INDEX_REWRITE_JOURNAL_LIMIT = 1000;
    static final int INDEX_REWRITE_TIME_LIMIT_MS = 1800000;
    static final int MAX_NAME_LENGTH = 254;
    static final String INDEX_FILE = "~index.dat";
    static final String BAK_INDEX_FILE = "~index.bak";
    static final String TMP_INDEX_FILE = "~index.tmp";
    static final String JOURNAL_FILE = "~journal.dat";
    static final String BAK_JOURNAL_FILE = "~journal.bak";
    static final String REBUILD_ERROR_FILE = "~rebuild.err";
    static final String INDEX_LOCK_FILE = "~index.lock";
    private static final String INDEX_VERSION_PREFIX = "VERSION=";
    private static final String MD5_PREFIX = "MD5:";
    private static final String NEXT_FILE_INDEX_ID_PREFIX = "NEXT-ID:";
    protected static final String INDEX_ITEM_INDENT = "  ";
    protected static final String INDEX_ITEM_SEPARATOR = ":";
    private final File indexFile;
    private final File journalFile;
    IndexJournal indexJournal;
    private int journalCount;
    private long nextFileIndexID = 0L;
    private Folder rootFolder;
    private boolean emptyFilesystem;
    private GhidraSwinglessTimer indexRewriteTimer;
    private PrintWriter rebuildErrWriter;

    IndexedLocalFileSystem(String rootPath, boolean isVersioned, boolean readOnly, boolean enableAsyncronousDispatching, boolean create) throws IOException {
        super(rootPath, isVersioned, readOnly, enableAsyncronousDispatching);
        this.indexFile = new File(rootPath, INDEX_FILE);
        this.journalFile = new File(rootPath, JOURNAL_FILE);
        if (create) {
            this.rootFolder = new Folder(this);
            if (readOnly) {
                this.emptyFilesystem = true;
                return;
            }
            if (this.indexFile.getParentFile().list().length != 0) {
                throw new IOException("data directory is not empty: " + rootPath);
            }
            this.writeIndex();
        } else {
            this.readIndex();
        }
        this.indexJournal = new IndexJournal();
        if (!readOnly) {
            this.indexRewriteTimer = new GhidraSwinglessTimer(1800000, () -> {
                IndexedLocalFileSystem indexedLocalFileSystem = this;
                synchronized (indexedLocalFileSystem) {
                    if (this.journalCount != 0) {
                        this.flushIndex();
                    }
                }
            });
            this.indexRewriteTimer.start();
            if (!create) {
                this.cleanupAfterConstruction();
            }
        }
    }

    protected IndexedLocalFileSystem(String rootPath) throws IOException {
        super(rootPath, false, false, false);
        this.indexFile = new File(rootPath, INDEX_FILE);
        this.indexFile.delete();
        this.journalFile = new File(rootPath, JOURNAL_FILE);
        this.journalFile.delete();
        this.rootFolder = new Folder(this);
        this.writeIndex();
        this.indexJournal = new IndexJournal();
    }

    @Override
    public int getMaxNameLength() {
        return 254;
    }

    private void refreshReadOnlyIndex() throws IOException {
        if (this.emptyFilesystem) {
            return;
        }
        this.readIndex();
        this.indexJournal = new IndexJournal();
    }

    @Override
    public LocalFolderItem[] getItems(String folderPath) throws IOException {
        String[] itemNames = this.getItemNames(folderPath, false);
        LocalFolderItem[] folderItems = new LocalFolderItem[itemNames.length];
        for (int i = 0; i < itemNames.length; ++i) {
            LocalFolderItem item = this.getItem(folderPath, itemNames[i]);
            if (item == null && !this.readOnly) {
                Msg.warn((Object)this, (Object)("Removing missing folder item from filesystem index: " + LocalFileSystem.getPath(folderPath, itemNames[i])));
                this.itemDeleted(folderPath, itemNames[i]);
            }
            folderItems[i] = item;
        }
        return folderItems;
    }

    @Override
    public synchronized void dispose() {
        if (this.rootFolder == null) {
            return;
        }
        if (this.indexRewriteTimer != null) {
            this.indexRewriteTimer.stop();
            this.indexRewriteTimer = null;
        }
        if (!this.readOnly) {
            this.flushIndex();
        }
        this.dispose(this.rootFolder);
        this.rootFolder = null;
        super.dispose();
    }

    private void dispose(Folder folder) {
        for (Folder subfolder : folder.folders.values()) {
            this.dispose(subfolder);
        }
        folder.folders.clear();
        for (Item item : folder.items.values()) {
            item.parent = null;
        }
        folder.items.clear();
    }

    private synchronized String getNextStorageName() {
        String storageName = StringUtilities.pad((String)Long.toHexString(this.nextFileIndexID++), (char)'0', (int)8);
        this.getStorageDir(storageName);
        return storageName;
    }

    private synchronized void bumpNextFileIndexID(String storageName) throws BadStorageNameException {
        try {
            long id = NumericUtilities.parseHexLong((String)storageName);
            this.nextFileIndexID = Math.max(this.nextFileIndexID, id + 1L);
        }
        catch (Exception e) {
            throw new BadStorageNameException(storageName);
        }
    }

    private File getStorageDir(String storageName) {
        int len = storageName.length();
        File dir = new File(this.root, storageName.substring(len - 3, len - 1));
        if (!dir.exists()) {
            dir.mkdir();
        }
        return dir;
    }

    private void flushIndex() {
        try {
            this.writeIndex();
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Failed to flush index file", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeIndex() throws IOException {
        if (this.readOnly) {
            throw new IOException("Unexpected attempt to write index for read-only filesystem");
        }
        MessageDigest messageDigest = null;
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e.toString());
        }
        File tempIndexFile = new File(this.root, TMP_INDEX_FILE);
        tempIndexFile.delete();
        try (PrintWriter indexWriter = new PrintWriter(tempIndexFile, "UTF8");){
            int version = this.getIndexImplementationVersion();
            if (version != 0) {
                String versionLine = INDEX_VERSION_PREFIX + version;
                this.digest(versionLine, messageDigest);
                indexWriter.println(versionLine);
            }
            this.writeIndexFolder(indexWriter, this.rootFolder, messageDigest);
            String idLine = NEXT_FILE_INDEX_ID_PREFIX + Long.toHexString(this.nextFileIndexID);
            this.digest(idLine, messageDigest);
            indexWriter.println(idLine);
            indexWriter.println(MD5_PREFIX + NumericUtilities.convertBytesToString((byte[])messageDigest.digest()));
            indexWriter.flush();
            indexWriter.close();
            if (indexWriter.checkError()) {
                indexWriter = null;
                throw new IOException("error occurred while writing filesystem index: " + String.valueOf(tempIndexFile));
            }
        }
        File backupIndexFile = new File(this.root, BAK_INDEX_FILE);
        File backupJournalFile = new File(this.root, BAK_JOURNAL_FILE);
        if (this.indexFile.exists()) {
            backupIndexFile.delete();
            backupJournalFile.delete();
            if (!this.indexFile.renameTo(backupIndexFile)) {
                throw new IOException("failed to backup filesystem index: " + String.valueOf(this.indexFile));
            }
            if (this.journalFile.exists() && !this.journalFile.renameTo(backupJournalFile)) {
                backupIndexFile.renameTo(this.indexFile);
                throw new IOException("failed to backup filesystem journal: " + String.valueOf(this.journalFile));
            }
        } else {
            backupIndexFile.delete();
            backupJournalFile.delete();
            backupIndexFile = null;
            backupJournalFile = null;
        }
        if (!tempIndexFile.renameTo(this.indexFile)) {
            if (backupIndexFile != null) {
                backupIndexFile.renameTo(this.indexFile);
            }
            if (backupJournalFile != null) {
                backupJournalFile.renameTo(this.journalFile);
            }
            throw new IOException("failed to update filesystem index (2): " + String.valueOf(this.indexFile));
        }
        this.journalCount = 0;
    }

    void digest(String str, MessageDigest messageDigest) {
        messageDigest.digest(str.getBytes());
    }

    private void writeIndexFolder(PrintWriter indexWriter, Folder folder, MessageDigest messageDigest) {
        String folderPath = folder.getPathname();
        indexWriter.println(folderPath);
        this.digest(folderPath, messageDigest);
        for (Item item : folder.items.values()) {
            String entry = this.formatIndexItem(item);
            indexWriter.println(INDEX_ITEM_INDENT + entry);
            this.digest(entry, messageDigest);
        }
        for (Folder subfolder : folder.folders.values()) {
            this.writeIndexFolder(indexWriter, subfolder, messageDigest);
        }
    }

    String formatIndexItem(Item item) {
        return item.getStorageName() + INDEX_ITEM_SEPARATOR + item.getName();
    }

    public int getIndexImplementationVersion() {
        return 0;
    }

    private boolean checkIndexVersion(String firstIndexLine) throws IndexVersionException {
        boolean consumed = firstIndexLine != null && firstIndexLine.startsWith(INDEX_VERSION_PREFIX);
        int indexImplVersion = this.getIndexImplementationVersion();
        int indexVersion = IndexedLocalFileSystem.getIndexVersion(firstIndexLine);
        if (indexVersion >= 0 && indexVersion < indexImplVersion) {
            throw new IndexVersionException("Filesystem Index upgrade/rebuild required", true);
        }
        if (indexVersion != indexImplVersion) {
            throw new IndexVersionException("Unsupported Filesystem Index version (newer version of application required)", false);
        }
        return consumed;
    }

    private static int getIndexVersion(String firstIndexLine) {
        int indexVersion = -1;
        if (firstIndexLine != null && firstIndexLine.length() != 0) {
            if (firstIndexLine.startsWith(INDEX_VERSION_PREFIX)) {
                String versionStr = firstIndexLine.substring(INDEX_VERSION_PREFIX.length());
                try {
                    indexVersion = Integer.parseInt(versionStr);
                }
                catch (Exception e) {
                    Msg.error(IndexedLocalFileSystem.class, (Object)("Invalid file-system version (" + versionStr + ")"));
                    indexVersion = Integer.MAX_VALUE;
                }
            } else {
                indexVersion = 0;
            }
        }
        return indexVersion;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int readIndexVersion(String rootPath) throws IOException {
        File indexFile = new File(rootPath, INDEX_FILE);
        BufferedReader indexReader = null;
        try {
            indexReader = new BufferedReader(new InputStreamReader((InputStream)new BufferedInputStream(new FileInputStream(indexFile)), "UTF8"));
            int n = IndexedLocalFileSystem.getIndexVersion(indexReader.readLine());
            return n;
        }
        finally {
            if (indexReader != null) {
                try {
                    indexReader.close();
                }
                catch (IOException iOException) {}
            }
        }
    }

    private void readIndex() throws IndexReadException {
        MessageDigest messageDigest = null;
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e.toString());
        }
        if (this.rootFolder != null) {
            this.dispose(this.rootFolder);
        }
        this.rootFolder = new Folder(this);
        String md5Str = null;
        String idLine = null;
        BufferedReader indexReader = null;
        try {
            indexReader = new BufferedReader(new InputStreamReader((InputStream)new BufferedInputStream(new FileInputStream(this.indexFile)), "UTF8"));
            String line = indexReader.readLine();
            if (this.checkIndexVersion(line)) {
                line = indexReader.readLine();
            }
            Folder currentFolder = null;
            while (line != null) {
                if (line.startsWith(MD5_PREFIX)) {
                    md5Str = line.substring(MD5_PREFIX.length());
                } else if (line.startsWith(NEXT_FILE_INDEX_ID_PREFIX)) {
                    md5Str = null;
                    this.digest(line, messageDigest);
                    idLine = line.substring(NEXT_FILE_INDEX_ID_PREFIX.length());
                } else {
                    md5Str = null;
                    idLine = null;
                    if (line.startsWith(SEPARATOR)) {
                        this.digest(line, messageDigest);
                        currentFolder = this.getFolder(line, GetFolderOption.CREATE);
                    } else {
                        String entry = line.substring(INDEX_ITEM_INDENT.length());
                        this.digest(entry, messageDigest);
                        if (this.parseIndexItem(currentFolder, entry) == null) {
                            throw new IOException("Invalid filesystem index: " + String.valueOf(this.indexFile));
                        }
                    }
                }
                line = indexReader.readLine();
            }
        }
        catch (Exception e) {
            if (e instanceof IndexVersionException) {
                throw (IndexVersionException)e;
            }
            throw new IndexReadException("Filesystem Index error: " + String.valueOf(this.indexFile), e);
        }
        finally {
            if (indexReader != null) {
                try {
                    indexReader.close();
                }
                catch (IOException iOException) {}
            }
        }
        try {
            this.nextFileIndexID = NumericUtilities.parseHexLong(idLine);
        }
        catch (Exception e) {
            throw new IndexReadException("Invalid Filesystem Index (NEXT-ID): " + String.valueOf(this.indexFile));
        }
        String md5Digest = NumericUtilities.convertBytesToString((byte[])messageDigest.digest());
        if (md5Str == null || !md5Str.equals(md5Digest)) {
            throw new IndexReadException("Invalid Filesystem Index (MD5): " + String.valueOf(this.indexFile));
        }
    }

    Item parseIndexItem(Folder parent, String entry) {
        int index = entry.indexOf(INDEX_ITEM_SEPARATOR);
        if (index < 0) {
            return null;
        }
        String storageName = entry.substring(0, index);
        String name = entry.substring(index + 1);
        return new Item(parent, name, storageName);
    }

    private void logRebuildError(String text) throws IOException {
        if (this.rebuildErrWriter == null) {
            File file = new File(this.root, REBUILD_ERROR_FILE);
            file.delete();
            this.rebuildErrWriter = new PrintWriter(new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file))));
        }
        this.rebuildErrWriter.println(text);
    }

    boolean rebuildIndex() throws IOException {
        for (File f : this.root.listFiles()) {
            if (!f.isDirectory()) continue;
            this.rebuildDirectoryIndex(f);
        }
        if (this.rebuildErrWriter != null) {
            this.rebuildErrWriter.close();
            return false;
        }
        return true;
    }

    private void rebuildDirectoryIndex(File dir) throws IOException {
        for (File f : dir.listFiles()) {
            if (!f.isFile() || !f.getName().endsWith(".prp")) continue;
            try {
                if (this.addFileToIndex(new IndexedPropertyFile(f))) continue;
                this.logRebuildError("Invalid item property file: " + String.valueOf(f));
            }
            catch (NotFoundException e) {
                this.logRebuildError("Item property file contains invalid parent path:" + String.valueOf(f));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addFileToIndex(ItemPropertyFile pfile) throws IOException, NotFoundException {
        String parentPath = pfile.getParentPath();
        String name = pfile.getName();
        if (parentPath == null || name == null) {
            return false;
        }
        String conflictedItemStorageName = this.findItemStorageName(parentPath, name);
        String storageName = pfile.getStorageName();
        if (conflictedItemStorageName != null) {
            try {
                if (storageName.compareTo(conflictedItemStorageName) <= 0) {
                    conflictedItemStorageName = storageName;
                    boolean bl = true;
                    return bl;
                }
                this.deallocateItemStorage(parentPath, name);
            }
            finally {
                Msg.warn((Object)this, (Object)("Detected orphaned project file " + conflictedItemStorageName + ": " + IndexedLocalFileSystem.getPath(parentPath, name)));
            }
        }
        this.indexJournal.open();
        try {
            Folder folder = this.addFolderToIndexIfMissing(parentPath);
            Item item = new Item(folder, name, pfile.getStorageName());
            this.bumpNextFileIndexID(item.getStorageName());
            this.indexJournal.addItem(item);
        }
        finally {
            this.indexJournal.close();
        }
        return true;
    }

    private Folder addFolderToIndexIfMissing(String folderPath) throws IOException, NotFoundException {
        if (SEPARATOR.equals(folderPath)) {
            return this.rootFolder;
        }
        int index = folderPath.lastIndexOf(SEPARATOR);
        String name = folderPath.substring(index + 1);
        String parentPath = index == 0 ? SEPARATOR : folderPath.substring(0, index);
        Folder parent = this.getFolder(parentPath, GetFolderOption.CREATE_ALL);
        Folder folder = parent.folders.get(name);
        if (folder != null) {
            return folder;
        }
        folder = new Folder(this);
        folder.parent = parent;
        folder.name = name;
        parent.folders.put(name, folder);
        this.indexJournal.addFolder(folderPath);
        return folder;
    }

    static int verifyIndexedFileStructure(File root) throws IndexReadException {
        int itemCount = 0;
        for (File f : root.listFiles()) {
            if (f.isDirectory()) {
                if (IndexedLocalFileSystem.isHiddenDirName(f.getName())) continue;
                itemCount += IndexedLocalFileSystem.verifyIndexedDirectory(f);
                continue;
            }
            String fname = f.getName();
            if (!fname.endsWith(".prp")) continue;
            throw new IndexReadException("Unexpected property file in filesystem root: " + fname);
        }
        return itemCount;
    }

    private static int verifyIndexedDirectory(File dir) throws IndexReadException {
        int itemCount = 0;
        String fname = dir.getName();
        boolean badFolder = true;
        if (fname.length() == 2) {
            try {
                Integer.parseInt(fname, 16);
                badFolder = false;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if (badFolder) {
            throw new IndexReadException("Unexpected folder in filesystem root: " + fname);
        }
        for (File f : dir.listFiles()) {
            fname = f.getName();
            if (f.isDirectory()) {
                if (IndexedLocalFileSystem.isHiddenDirName(fname)) continue;
                throw new IndexReadException("Unexpected folder in filesystem: " + dir.getName() + SEPARATOR + fname);
            }
            if (!fname.endsWith(".prp")) continue;
            IndexedLocalFileSystem.verifyIndexedPropertyFile(f);
            ++itemCount;
        }
        return itemCount;
    }

    private static void verifyIndexedPropertyFile(File f) throws IndexReadException {
        String fname = f.getName();
        fname = fname.substring(0, fname.length() - ".prp".length());
        boolean badFile = true;
        int len = fname.length();
        if (len >= 8 && len <= 16) {
            try {
                Integer.parseInt(fname, 16);
                badFile = false;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if (badFile) {
            throw new IndexReadException("Unexpected property file in filesystem: " + f.getParentFile().getName() + SEPARATOR + fname);
        }
    }

    Folder getFolder(String path, GetFolderOption option) throws NotFoundException {
        if (this.rootFolder == null) {
            throw new NotFoundException("Filesystem has been disposed");
        }
        if (!path.startsWith(SEPARATOR)) {
            throw new NotFoundException("Invalid folder path: " + path);
        }
        Folder folder = this.rootFolder;
        if (path.length() == 1) {
            return folder;
        }
        String[] names = path.substring(1).split(SEPARATOR);
        for (int i = 0; i < names.length; ++i) {
            String name = names[i];
            Folder subfolder = folder.folders.get(name);
            if (subfolder == null) {
                if (option == GetFolderOption.CREATE && i == names.length - 1 || option == GetFolderOption.CREATE_ALL_NOTIFY || option == GetFolderOption.CREATE_ALL) {
                    subfolder = new Folder(this);
                    subfolder.parent = folder;
                    subfolder.name = name;
                    folder.folders.put(name, subfolder);
                    if (option == GetFolderOption.CREATE_ALL_NOTIFY) {
                        this.eventManager.folderCreated(folder.getPathname(), name);
                    }
                } else {
                    throw new NotFoundException("Folder not found: " + IndexedLocalFileSystem.getPath(folder.getPathname(), name));
                }
            }
            folder = subfolder;
        }
        return folder;
    }

    private String findItemStorageName(String folderPath, String name) {
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            Item item = folder.items.get(name);
            if (item != null) {
                return item.itemStorage.storageName;
            }
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
        return null;
    }

    @Override
    protected LocalFileSystem.ItemStorage findItemStorage(String folderPath, String itemName) throws FileNotFoundException {
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            Item item = folder.items.get(itemName);
            if (item != null) {
                return item.itemStorage;
            }
        }
        catch (NotFoundException notFoundException) {
            // empty catch block
        }
        throw new FileNotFoundException("Item not found: " + IndexedLocalFileSystem.getPath(folderPath, itemName));
    }

    @Override
    protected LocalFileSystem.ItemStorage allocateItemStorage(String folderPath, String itemName) throws IOException, InvalidNameException {
        this.indexJournal.open();
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.CREATE_ALL_NOTIFY);
            if (folder.items.containsKey(itemName)) {
                throw new DuplicateFileException(IndexedLocalFileSystem.getPath(folderPath, itemName) + " already exists.");
            }
            Item item = new Item(folder, itemName);
            this.indexJournal.addItem(item);
            LocalFileSystem.ItemStorage itemStorage = item.itemStorage;
            return itemStorage;
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException("Folder not found: " + folderPath);
        }
        finally {
            this.indexJournal.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void deallocateItemStorage(String folderPath, String itemName) throws IOException {
        this.indexJournal.open();
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            if (folder == null) {
                return;
            }
            Item item = folder.items.get(itemName);
            if (item != null) {
                this.indexJournal.deleteItem(item);
                folder.items.remove(itemName);
            }
        }
        catch (NotFoundException notFoundException) {
        }
        finally {
            this.indexJournal.close();
        }
    }

    @Override
    protected synchronized void itemDeleted(String folderPath, String itemName) throws IOException {
        this.deallocateItemStorage(folderPath, itemName);
        super.itemDeleted(folderPath, itemName);
    }

    void mapFileID(String fileId, Item item) {
    }

    void unmapFileID(String fileId) {
    }

    @Override
    public String[] getItemNames(String folderPath, boolean includeHiddenFiles) throws IOException {
        if (this.readOnly) {
            this.refreshReadOnlyIndex();
        }
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            Set<String> nameSet = folder.items.keySet();
            int count = nameSet.size();
            ArrayList<String> fileList = new ArrayList<String>(count);
            for (String name : nameSet) {
                if (!includeHiddenFiles && name.startsWith(".ghidra.")) continue;
                fileList.add(name);
            }
            return fileList.toArray(new String[fileList.size()]);
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException("Folder not found: " + folderPath);
        }
    }

    @Override
    public int getItemCount() throws IOException {
        this.checkDisposed();
        if (this.readOnly) {
            this.refreshReadOnlyIndex();
        }
        return this.getItemCount(this.rootFolder);
    }

    private int getItemCount(Folder folder) {
        int count = folder.items.size();
        for (Folder f : folder.folders.values()) {
            count += this.getItemCount(f);
        }
        return count;
    }

    @Override
    public synchronized String[] getFolderNames(String folderPath) throws IOException {
        this.checkDisposed();
        if (this.readOnly) {
            this.refreshReadOnlyIndex();
        }
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            Set<String> nameSet = folder.folders.keySet();
            return nameSet.toArray(new String[nameSet.size()]);
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException("Folder not found: " + folderPath);
        }
    }

    @Override
    public synchronized void createFolder(String parentPath, String folderName) throws InvalidNameException, IOException {
        this.checkDisposed();
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        this.testValidName(parentPath, true);
        this.testValidName(folderName, false);
        String path = IndexedLocalFileSystem.getPath(parentPath, folderName);
        this.indexJournal.open();
        try {
            Folder parent = this.getFolder(parentPath, GetFolderOption.CREATE_ALL_NOTIFY);
            if (parent.folders.get(folderName) != null) {
                return;
            }
            this.indexJournal.addFolder(path);
            Folder folder = new Folder(this);
            folder.parent = parent;
            folder.name = folderName;
            parent.folders.put(folderName, folder);
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException("Folder not found: " + parentPath);
        }
        finally {
            this.indexJournal.close();
        }
        this.eventManager.folderCreated(parentPath, IndexedLocalFileSystem.getName(path));
    }

    @Override
    public synchronized void deleteFolder(String folderPath) throws IOException {
        this.checkDisposed();
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        if (SEPARATOR.equals(folderPath)) {
            throw new IOException("Root folder may not be deleted");
        }
        this.indexJournal.open();
        try {
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            if (folder.folders.size() != 0 || folder.items.size() != 0) {
                throw new FolderNotEmptyException(folderPath + " is not empty");
            }
            this.indexJournal.deleteFolder(folderPath);
            folder.parent.folders.remove(folder.name);
        }
        catch (NotFoundException e) {
            return;
        }
        finally {
            this.indexJournal.close();
        }
        this.eventManager.folderDeleted(IndexedLocalFileSystem.getParentPath(folderPath), IndexedLocalFileSystem.getName(folderPath));
    }

    void migrateItem(LocalFolderItem item) throws IOException {
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        String destFolderPath = item.getParentPath();
        String itemName = item.getName();
        String path = item.getPathName();
        this.indexJournal.open();
        try {
            Folder folder = this.getFolder(destFolderPath, GetFolderOption.READ_ONLY);
            if (folder.items.containsKey(itemName)) {
                throw new DuplicateFileException("Item already exists: " + path);
            }
            Item newItem = new Item(folder, itemName);
            item.moveTo(newItem.itemStorage.dir, newItem.itemStorage.storageName, path, itemName);
            newItem.itemStorage.getPropertyFile().writeState();
            this.indexJournal.addItem(newItem);
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException(e.getMessage());
        }
        finally {
            this.indexJournal.close();
        }
        this.getListener().itemCreated(destFolderPath, itemName);
    }

    @Override
    public synchronized void moveItem(String folderPath, String name, String newFolderPath, String newName) throws IOException, InvalidNameException {
        this.checkDisposed();
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        String oldPath = IndexedLocalFileSystem.getPath(folderPath, name);
        String newPath = IndexedLocalFileSystem.getPath(newFolderPath, newName);
        boolean success = false;
        this.indexJournal.open();
        try {
            LocalFolderItem conflictFolderItem;
            Folder folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            Item item = folder.items.get(name);
            if (item == null) {
                throw new FileNotFoundException("Item not found: " + oldPath);
            }
            if (folderPath.equals(newFolderPath) && name.equals(newName)) {
                return;
            }
            this.testValidName(newFolderPath, true);
            this.testValidName(newName, false);
            LocalFolderItem folderItem = this.getItem(folderPath, name);
            if (folderItem == null) {
                throw new FileNotFoundException("Item not found: " + oldPath);
            }
            folderItem.checkInUse();
            Folder newFolder = folder;
            if (!folderPath.equals(newFolderPath)) {
                newFolder = this.getFolder(newFolderPath, GetFolderOption.CREATE_ALL_NOTIFY);
            }
            if ((conflictFolderItem = this.getItem(newFolderPath, newName)) != null) {
                throw new DuplicateFileException("Item already exists: " + newName);
            }
            folderItem.moveTo(item.itemStorage.dir, item.itemStorage.storageName, newFolderPath, newName);
            folder.items.remove(name);
            item.parent = newFolder;
            item.itemStorage.itemName = newName;
            item.itemStorage.folderPath = newFolderPath;
            newFolder.items.put(newName, item);
            this.indexJournal.moveItem(oldPath, newPath);
            success = true;
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException(e.getMessage());
        }
        finally {
            this.indexJournal.close();
            if (!success) {
                this.deleteEmptyVersionedFolders(newFolderPath);
            }
        }
        if (folderPath.equals(newFolderPath)) {
            this.eventManager.itemRenamed(folderPath, name, newName);
        } else {
            this.eventManager.itemMoved(folderPath, name, newFolderPath, newName);
        }
        this.deleteEmptyVersionedFolders(folderPath);
    }

    @Override
    public synchronized void moveFolder(String parentPath, String folderName, String newParentPath) throws InvalidNameException, IOException {
        Folder folder;
        this.checkDisposed();
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        this.testValidName(newParentPath, true);
        String folderPath = IndexedLocalFileSystem.getPath(parentPath, folderName);
        boolean success = false;
        this.indexJournal.open();
        try {
            folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            String newFolderPath = IndexedLocalFileSystem.getPath(newParentPath, folderName);
            Folder newParentFolder = this.getFolder(newParentPath, GetFolderOption.CREATE_ALL_NOTIFY);
            if (newParentFolder.folders.get(folderName) != null) {
                throw new DuplicateFileException(newFolderPath + " already exists.");
            }
            this.indexJournal.moveFolder(folderPath, newFolderPath);
            folder.parent.folders.remove(folderName);
            folder.parent = newParentFolder;
            newParentFolder.folders.put(folderName, folder);
            success = true;
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException(e.getMessage());
        }
        finally {
            this.indexJournal.close();
            if (!success) {
                this.deleteEmptyVersionedFolders(newParentPath);
            }
        }
        this.updateAffectedItemPaths(folder);
        this.eventManager.folderMoved(parentPath, folderName, newParentPath);
        this.deleteEmptyVersionedFolders(parentPath);
    }

    private void updateAffectedItemPaths(Folder folder) throws IOException {
        String newFolderPath = folder.getPathname();
        for (Item item : folder.items.values()) {
            LocalFileSystem.ItemStorage itemStorage = item.itemStorage;
            ItemPropertyFile pfile = item.itemStorage.getPropertyFile();
            pfile.moveTo(itemStorage.dir, itemStorage.storageName, newFolderPath, itemStorage.itemName);
            itemStorage.folderPath = newFolderPath;
        }
        for (Folder subfolder : folder.folders.values()) {
            this.updateAffectedItemPaths(subfolder);
        }
    }

    @Override
    public synchronized void renameFolder(String parentPath, String folderName, String newFolderName) throws InvalidNameException, IOException {
        Folder folder;
        this.checkDisposed();
        if (this.readOnly) {
            throw new ReadOnlyException();
        }
        this.testValidName(newFolderName, false);
        String folderPath = IndexedLocalFileSystem.getPath(parentPath, folderName);
        this.indexJournal.open();
        try {
            folder = this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            if (folder.parent.folders.get(newFolderName) != null) {
                throw new DuplicateFileException(IndexedLocalFileSystem.getPath(parentPath, newFolderName) + " already exists.");
            }
            this.indexJournal.moveFolder(folderPath, IndexedLocalFileSystem.getPath(parentPath, newFolderName));
            folder.parent.folders.remove(folderName);
            folder.name = newFolderName;
            folder.parent.folders.put(newFolderName, folder);
        }
        catch (NotFoundException e) {
            throw new FileNotFoundException(folderPath + " does not exist or is not a folder");
        }
        finally {
            this.indexJournal.close();
        }
        this.updateAffectedItemPaths(folder);
        this.eventManager.folderRenamed(parentPath, folderName, newFolderName);
    }

    @Override
    public synchronized boolean folderExists(String folderPath) {
        try {
            this.checkDisposed();
            this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            return true;
        }
        catch (NotFoundException | IOException e) {
            return false;
        }
    }

    public static boolean isIndexed(String rootPath) {
        File rootFile = new File(rootPath);
        if (!rootFile.isDirectory()) {
            return false;
        }
        File indexFile = new File(rootPath, INDEX_FILE);
        return indexFile.exists();
    }

    public static boolean hasIndexedStructure(String rootPath) {
        File rootFile = new File(rootPath);
        if (rootFile.isDirectory()) {
            try {
                int itemCount = IndexedLocalFileSystem.verifyIndexedFileStructure(rootFile);
                return itemCount != 0;
            }
            catch (IndexReadException indexReadException) {
                // empty catch block
            }
        }
        return false;
    }

    static IndexedLocalFileSystem getFileSystem(String rootPath, boolean isVersioned, boolean readOnly, boolean enableAsyncronousDispatching) throws IOException {
        try {
            return new IndexedLocalFileSystem(rootPath, isVersioned, readOnly, enableAsyncronousDispatching, false);
        }
        catch (IndexReadException e) {
            if (readOnly) {
                throw e;
            }
            Msg.error(LocalFileSystem.class, (Object)("Indexed filesystem error: " + e.getMessage()));
            Msg.info(LocalFileSystem.class, (Object)("Attempting index rebuild: " + rootPath));
            if (!IndexedLocalFileSystem.rebuild(new File(rootPath))) {
                throw e;
            }
            return new IndexedLocalFileSystem(rootPath, isVersioned, readOnly, enableAsyncronousDispatching, false);
        }
    }

    public static boolean rebuild(File rootDir) throws IOException {
        IndexedLocalFileSystem.verifyIndexedFileStructure(rootDir);
        IndexedLocalFileSystem fs = new IndexedLocalFileSystem(rootDir.getAbsolutePath());
        fs.rebuildIndex();
        fs.cleanupAfterConstruction();
        fs.dispose();
        File errorFile = new File(rootDir, REBUILD_ERROR_FILE);
        if (errorFile.exists()) {
            Msg.error(LocalFileSystem.class, (Object)("Indexed filesystem rebuild failed, see log for details: " + String.valueOf(errorFile)));
            return false;
        }
        Msg.info(LocalFileSystem.class, (Object)("Index rebuild completed: " + String.valueOf(rootDir)));
        return true;
    }

    class Folder {
        Folder parent;
        String name;
        Map<String, Item> items = new TreeMap<String, Item>();
        Map<String, Folder> folders = new TreeMap<String, Folder>();

        Folder(IndexedLocalFileSystem this$0) {
        }

        public String getPathname() {
            if (this.parent == null) {
                return FileSystem.SEPARATOR;
            }
            return LocalFileSystem.getPath(this.parent.getPathname(), this.name);
        }

        public String toString() {
            StringBuffer buf = new StringBuffer();
            String path = this.getPathname();
            Folder p = this.parent;
            while (p != null) {
                buf.append(' ');
                p = p.parent;
            }
            String pad = buf.toString();
            buf.append(path);
            for (Item item : this.items.values()) {
                buf.append('\n');
                buf.append(pad);
                buf.append(' ');
                buf.append(item.toString());
                if (item.parent == this) continue;
                buf.append(" **BAD-PARENT**");
            }
            for (Folder sf : this.folders.values()) {
                buf.append('\n');
                buf.append(sf.toString());
            }
            return buf.toString();
        }
    }

    class IndexJournal {
        private PrintWriter journalWriter;

        IndexJournal() throws IOException {
            if (IndexedLocalFileSystem.this.journalFile.exists()) {
                this.replayJournal();
            }
        }

        void close() {
            if (this.journalWriter != null) {
                this.journalWriter.close();
                this.journalWriter = null;
            }
            if (IndexedLocalFileSystem.this.journalCount >= 1000) {
                IndexedLocalFileSystem.this.flushIndex();
            }
        }

        void open() throws IOException {
            if (IndexedLocalFileSystem.this.readOnly) {
                throw new ReadOnlyException();
            }
            this.journalWriter = new PrintWriter(new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(IndexedLocalFileSystem.this.journalFile, true), "UTF8")));
        }

        private void replayJournal() throws IndexReadException {
            int lineNum = 0;
            BufferedReader journalReader = null;
            try {
                String line;
                journalReader = new BufferedReader(new InputStreamReader((InputStream)new BufferedInputStream(new FileInputStream(IndexedLocalFileSystem.this.journalFile)), "UTF8"));
                while ((line = journalReader.readLine()) != null) {
                    ++lineNum;
                    String[] args = line.split(IndexedLocalFileSystem.INDEX_ITEM_SEPARATOR);
                    if ("FADD".equals(args[0])) {
                        this.replayFolderAdd(args[1]);
                        continue;
                    }
                    if ("FDEL".equals(args[0])) {
                        this.replayFolderDelete(args[1]);
                        continue;
                    }
                    if ("FMV".equals(args[0])) {
                        this.replayFolderMove(args[1], args[2]);
                        continue;
                    }
                    if ("IADD".equals(args[0])) {
                        this.replayItemAdd(args[1], args[2]);
                        continue;
                    }
                    if ("IDEL".equals(args[0])) {
                        this.replayItemDelete(args[1]);
                        continue;
                    }
                    if ("IMV".equals(args[0])) {
                        this.replayItemMove(args[1], args[2]);
                        continue;
                    }
                    if ("IDSET".equals(args[0])) {
                        this.replayFileIdSet(args[1], args[2]);
                        continue;
                    }
                    throw new IndexReadException("Invalid index journal (" + lineNum + ") - unable to replay: " + String.valueOf(IndexedLocalFileSystem.this.journalFile));
                }
                journalReader.close();
                journalReader = null;
                IndexedLocalFileSystem.this.journalCount = lineNum;
                if (!IndexedLocalFileSystem.this.readOnly) {
                    IndexedLocalFileSystem.this.writeIndex();
                }
            }
            catch (Exception e) {
                if (e instanceof IndexReadException) {
                    throw (IndexReadException)e;
                }
                throw new IndexReadException("Index journal error (" + lineNum + ") - unable to replay: " + String.valueOf(IndexedLocalFileSystem.this.journalFile), e);
            }
            finally {
                if (journalReader != null) {
                    try {
                        journalReader.close();
                    }
                    catch (IOException iOException) {}
                }
            }
        }

        private void replayFolderAdd(String folderPath) throws NotFoundException, DuplicateFileException {
            int index = folderPath.lastIndexOf(FileSystem.SEPARATOR);
            String name = folderPath.substring(index + 1);
            String parentPath = index == 0 ? FileSystem.SEPARATOR : folderPath.substring(0, index);
            Folder parent = IndexedLocalFileSystem.this.getFolder(parentPath, GetFolderOption.CREATE_ALL);
            if (parent.folders.get(name) != null) {
                throw new DuplicateFileException("Folder already exists: " + folderPath);
            }
            Folder folder = new Folder(IndexedLocalFileSystem.this);
            folder.parent = parent;
            folder.name = name;
            parent.folders.put(name, folder);
        }

        private void replayFolderDelete(String folderPath) throws NotFoundException {
            Folder folder = IndexedLocalFileSystem.this.getFolder(folderPath, GetFolderOption.READ_ONLY);
            folder.parent.folders.remove(folder.name);
        }

        private void replayFolderMove(String oldPath, String newPath) throws NotFoundException, DuplicateFileException {
            Folder folder = IndexedLocalFileSystem.this.getFolder(oldPath, GetFolderOption.READ_ONLY);
            int index = newPath.lastIndexOf(FileSystem.SEPARATOR);
            String newName = newPath.substring(index + 1);
            String newParentPath = index == 0 ? FileSystem.SEPARATOR : newPath.substring(0, index);
            Folder newParent = IndexedLocalFileSystem.this.getFolder(newParentPath, GetFolderOption.CREATE_ALL);
            if (newParent.folders.get(newName) != null) {
                throw new DuplicateFileException("Folder already exists: " + newPath);
            }
            folder.parent.folders.remove(folder.name);
            folder.name = newName;
            folder.parent = newParent;
            newParent.folders.put(newName, folder);
        }

        private void replayItemAdd(String storageName, String itemPath) throws NotFoundException, BadStorageNameException, DuplicateFileException {
            int index = itemPath.lastIndexOf(FileSystem.SEPARATOR);
            String name = itemPath.substring(index + 1);
            String parentPath = index == 0 ? FileSystem.SEPARATOR : itemPath.substring(0, index);
            Folder parent = IndexedLocalFileSystem.this.getFolder(parentPath, GetFolderOption.CREATE_ALL);
            if (parent.items.get(name) != null) {
                throw new DuplicateFileException("Item already exists: " + itemPath);
            }
            new Item(parent, name, storageName);
            IndexedLocalFileSystem.this.bumpNextFileIndexID(storageName);
        }

        private void replayItemDelete(String itemPath) throws NotFoundException {
            int index = itemPath.lastIndexOf(FileSystem.SEPARATOR);
            String name = itemPath.substring(index + 1);
            String parentPath = index == 0 ? FileSystem.SEPARATOR : itemPath.substring(0, index);
            Folder parent = IndexedLocalFileSystem.this.getFolder(parentPath, GetFolderOption.READ_ONLY);
            if (parent.items.remove(name) == null) {
                throw new NotFoundException("Item not found: " + itemPath);
            }
        }

        private void replayItemMove(String oldPath, String newPath) throws NotFoundException {
            int index = oldPath.lastIndexOf(FileSystem.SEPARATOR);
            String name = oldPath.substring(index + 1);
            String parentPath = index == 0 ? FileSystem.SEPARATOR : oldPath.substring(0, index);
            Folder parent = IndexedLocalFileSystem.this.getFolder(parentPath, GetFolderOption.READ_ONLY);
            Item item = parent.items.get(name);
            if (item == null) {
                throw new NotFoundException("Item not found: " + oldPath);
            }
            index = newPath.lastIndexOf(FileSystem.SEPARATOR);
            String newName = newPath.substring(index + 1);
            String newParentPath = index == 0 ? FileSystem.SEPARATOR : newPath.substring(0, index);
            Folder newParent = IndexedLocalFileSystem.this.getFolder(newParentPath, GetFolderOption.CREATE_ALL);
            if (newParent.items.get(newName) != null) {
                throw new NotFoundException("Item already exists: " + newPath);
            }
            parent.items.remove(name);
            item.set(newParent, newName, item.getFileID());
            newParent.items.put(newName, item);
        }

        private void replayFileIdSet(String path, String fileId) throws NotFoundException {
            int index = path.lastIndexOf(FileSystem.SEPARATOR);
            String name = path.substring(index + 1);
            String parentPath = index == 0 ? FileSystem.SEPARATOR : path.substring(0, index);
            Folder parent = IndexedLocalFileSystem.this.getFolder(parentPath, GetFolderOption.READ_ONLY);
            Item item = parent.items.get(name);
            if (item == null) {
                throw new NotFoundException("Item not found: " + path);
            }
            item.setFileID(fileId);
        }

        void addFolder(String folderPath) throws IOException {
            this.journalWriter.println("FADD:" + folderPath);
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void deleteFolder(String folderPath) throws IOException {
            this.journalWriter.println("FDEL:" + folderPath);
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void moveFolder(String oldPath, String newPath) throws IOException {
            this.journalWriter.println("FMV:" + oldPath + IndexedLocalFileSystem.INDEX_ITEM_SEPARATOR + newPath);
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void addItem(Item item) throws IOException {
            this.journalWriter.println("IADD:" + item.getStorageName() + IndexedLocalFileSystem.INDEX_ITEM_SEPARATOR + item.getPathname());
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void deleteItem(Item item) throws IOException {
            this.journalWriter.println("IDEL:" + item.getPathname());
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void moveItem(String oldPath, String newPath) throws IOException {
            this.journalWriter.println("IMV:" + oldPath + IndexedLocalFileSystem.INDEX_ITEM_SEPARATOR + newPath);
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }

        void fileIdSet(String path, String fileId) throws IOException {
            this.journalWriter.println("IDSET:" + path + IndexedLocalFileSystem.INDEX_ITEM_SEPARATOR + fileId);
            ++IndexedLocalFileSystem.this.journalCount;
            if (this.journalWriter.checkError()) {
                throw new IOException("Journal update error");
            }
        }
    }

    class Item {
        private String fileId;
        private Folder parent;
        private String storageName;
        LocalFileSystem.ItemStorage itemStorage;

        Item(Folder parent, String name) {
            this.storageName = IndexedLocalFileSystem.this.getNextStorageName();
            this.set(parent, name, null);
        }

        Item(Folder parent, String name, String fileId, String storageName) {
            this.storageName = storageName;
            this.set(parent, name, fileId);
        }

        Item(Folder parent, String name, String storageName) {
            this.storageName = storageName;
            this.set(parent, name);
        }

        void set(Folder newParent, String newName, String newFileId) {
            if (this.parent != null && this.itemStorage != null) {
                this.parent.items.remove(this.itemStorage.itemName);
            }
            this.parent = newParent;
            this.itemStorage = new IndexedItemStorage(IndexedLocalFileSystem.this.getStorageDir(this.storageName), this.storageName, this.parent.getPathname(), newName);
            this.parent.items.put(newName, this);
            this.setFileID(newFileId);
        }

        void set(Folder newParent, String newName) {
            if (this.parent != null && this.itemStorage != null) {
                this.parent.items.remove(this.itemStorage.itemName);
            }
            this.parent = newParent;
            this.itemStorage = new IndexedItemStorage(IndexedLocalFileSystem.this.getStorageDir(this.storageName), this.storageName, this.parent.getPathname(), newName);
            this.parent.items.put(newName, this);
            try {
                this.setFileID(this.itemStorage.getPropertyFile().getFileID());
            }
            catch (IOException e) {
                this.setFileID(null);
            }
        }

        String getName() {
            return this.itemStorage.itemName;
        }

        void setFileID(String newFileId) {
            if (this.fileId != null) {
                IndexedLocalFileSystem.this.unmapFileID(this.fileId);
            }
            this.fileId = newFileId;
            if (this.fileId != null) {
                IndexedLocalFileSystem.this.mapFileID(this.fileId, this);
            }
        }

        String getFileID() {
            return this.fileId;
        }

        String getStorageName() {
            return this.itemStorage.storageName;
        }

        String getPathname() {
            return LocalFileSystem.getPath(this.parent.getPathname(), this.itemStorage.itemName);
        }

        public String toString() {
            return this.itemStorage.toString();
        }
    }

    public static class BadStorageNameException
    extends IOException {
        private static final long serialVersionUID = 1L;

        BadStorageNameException(String storageName) {
            super("Bad item storage name (expected hex value): " + storageName);
        }
    }

    public static class IndexVersionException
    extends IndexReadException {
        private static final long serialVersionUID = 1L;
        boolean canUpgrade = false;

        IndexVersionException(String msg, boolean canUpgrade) {
            super(msg);
            this.canUpgrade = canUpgrade;
        }

        public boolean canUpgrade() {
            return this.canUpgrade;
        }
    }

    static enum GetFolderOption {
        READ_ONLY,
        CREATE,
        CREATE_ALL,
        CREATE_ALL_NOTIFY;

    }

    public static class IndexReadException
    extends IOException {
        private static final long serialVersionUID = 1L;

        IndexReadException(String msg) {
            super(msg);
        }

        IndexReadException(String msg, Throwable cause) {
            super(msg, cause);
        }
    }

    static class IndexedItemStorage
    extends LocalFileSystem.ItemStorage {
        IndexedItemStorage(File dir, String storageName, String folderPath, String itemName) {
            super(dir, storageName, folderPath, itemName);
        }

        @Override
        ItemPropertyFile getPropertyFile() throws IOException {
            return new IndexedPropertyFile(this.dir, this.storageName, this.folderPath, this.itemName);
        }
    }
}

