/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server.remote;

import generic.jar.ResourceFile;
import generic.random.SecureRandomFactory;
import ghidra.framework.Application;
import ghidra.framework.ApplicationConfiguration;
import ghidra.framework.remote.GhidraPrincipal;
import ghidra.framework.remote.GhidraServerHandle;
import ghidra.framework.remote.RemoteRepositoryServerHandle;
import ghidra.net.ApplicationKeyManagerFactory;
import ghidra.net.SSLContextInitializer;
import ghidra.server.RepositoryManager;
import ghidra.server.remote.GhidraSSLServerSocket;
import ghidra.server.remote.GhidraServerApplicationLayout;
import ghidra.server.remote.InetNameLookup;
import ghidra.server.remote.RepositoryServerHandleImpl;
import ghidra.server.remote.ServerPortFactory;
import ghidra.server.security.AnonymousAuthenticationModule;
import ghidra.server.security.AuthenticationModule;
import ghidra.server.security.JAASAuthenticationModule;
import ghidra.server.security.Krb5ActiveDirectoryAuthenticationModule;
import ghidra.server.security.PKIAuthenticationModule;
import ghidra.server.security.PasswordFileAuthenticationModule;
import ghidra.server.security.SSHAuthenticationModule;
import ghidra.server.stream.BlockStreamServer;
import ghidra.server.stream.RemoteBlockStreamHandle;
import ghidra.util.SystemUtilities;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.DuplicateNameException;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputFilter;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.rmi.NoSuchObjectException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.RMIClientSocketFactory;
import java.rmi.server.RMIServerSocketFactory;
import java.rmi.server.UnicastRemoteObject;
import java.security.cert.CertificateException;
import java.util.Enumeration;
import java.util.List;
import javax.rmi.ssl.SslRMIClientSocketFactory;
import javax.rmi.ssl.SslRMIServerSocketFactory;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.security.auth.x500.X500Principal;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;
import utility.application.ApplicationLayout;

public class GhidraServer
extends UnicastRemoteObject
implements GhidraServerHandle {
    private static final String SERIAL_FILTER_FILE = "serial.filter";
    private static final String TLS_SERVER_PROTOCOLS_PROPERTY = "ghidra.tls.server.protocols";
    private static final String TLS_ENABLED_CIPHERS_PROPERTY = "jdk.tls.server.cipherSuites";
    private static final String SERIALIZATION_FILTER_DISABLED_PROPERTY = "ghidra.server.serialization.filter.disabled";
    private static SslRMIServerSocketFactory serverSocketFactory;
    private static SslRMIClientSocketFactory clientSocketFactory;
    private static InetAddress bindAddress;
    private static Logger log;
    private static String HELP_FILE;
    private static String USAGE_ARGS;
    private static final String RMI_SERVER_PROPERTY = "java.rmi.server.hostname";
    private static GhidraServer server;
    private RepositoryManager mgr;
    private AuthenticationModule authModule;
    private SSHAuthenticationModule sshAuthModule;
    private AnonymousAuthenticationModule anonymousAuthModule;
    private BlockStreamServer blockStreamServer;
    private boolean autoProvisionAuthedUsers;
    private static final int IP_INTERFACE_RETRY_TIME_SEC = 5;
    private static final int IP_INTERFACE_MAX_RETRIES = 12;

    GhidraServer(File rootDir, AuthMode authMode, String loginDomain, boolean allowUserToSpecifyName, boolean altSSHLoginAllowed, int defaultPasswordExpirationDays, boolean allowAnonymousAccess, boolean autoProvisionAuthedUsers, File jaasConfigFile) throws IOException, CertificateException {
        super(ServerPortFactory.getRMISSLPort(), clientSocketFactory, serverSocketFactory);
        this.autoProvisionAuthedUsers = autoProvisionAuthedUsers;
        if (log == null) {
            log = LogManager.getLogger(GhidraServer.class);
        }
        if (allowAnonymousAccess) {
            this.anonymousAuthModule = new AnonymousAuthenticationModule();
        }
        boolean supportLocalPasswords = false;
        switch (authMode.ordinal()) {
            case 1: {
                supportLocalPasswords = true;
                this.authModule = new PasswordFileAuthenticationModule(allowUserToSpecifyName);
                break;
            }
            case 3: {
                if (altSSHLoginAllowed) {
                    log.warn("SSH authentication option ignored when PKI authentication used");
                    altSSHLoginAllowed = false;
                }
                SecureRandomFactory.getSecureRandom();
                this.authModule = new PKIAuthenticationModule(allowAnonymousAccess);
                break;
            }
            case 0: {
                if (!altSSHLoginAllowed) break;
                log.warn("SSH authentication option ignored when no authentication used");
                altSSHLoginAllowed = false;
                break;
            }
            case 4: {
                this.authModule = new JAASAuthenticationModule("auth", allowUserToSpecifyName, jaasConfigFile);
                break;
            }
            case 2: {
                if (loginDomain == null || loginDomain.isBlank()) {
                    throw new IllegalArgumentException("Missing login domain value -d<ad_domain>");
                }
                this.authModule = new Krb5ActiveDirectoryAuthenticationModule(loginDomain, allowUserToSpecifyName);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported Authentication mode: " + String.valueOf((Object)authMode));
            }
        }
        if (altSSHLoginAllowed) {
            SecureRandomFactory.getSecureRandom();
            this.sshAuthModule = new SSHAuthenticationModule(allowUserToSpecifyName);
        }
        this.mgr = new RepositoryManager(rootDir, supportLocalPasswords, defaultPasswordExpirationDays, allowAnonymousAccess);
        server = this;
        GhidraServer.setGlobalSerializationFilter();
        this.blockStreamServer = BlockStreamServer.getBlockStreamServer();
        ServerSocket streamServerSocket = serverSocketFactory != null ? serverSocketFactory.createServerSocket(ServerPortFactory.getStreamPort()) : new GhidraSSLServerSocket(ServerPortFactory.getStreamPort(), bindAddress, null, null, authMode == AuthMode.PKI_LOGIN);
        this.blockStreamServer.startServer(streamServerSocket, GhidraServer.initRemoteAccessHostname());
    }

    public Callback[] getAuthenticationCallbacks() throws RemoteException {
        log.info("Authentication callbacks requested by " + RepositoryManager.getRMIClient());
        try {
            Callback[] callbacks;
            Callback[] callbackArray = callbacks = this.authModule != null ? this.authModule.getAuthenticationCallbacks() : null;
            if (this.sshAuthModule != null) {
                callbacks = this.sshAuthModule.addAuthenticationCallbacks(callbacks);
            }
            if (this.anonymousAuthModule != null && (this.authModule == null || this.authModule.anonymousCallbacksAllowed())) {
                callbacks = this.anonymousAuthModule.addAuthenticationCallbacks(callbacks);
            }
            return callbacks;
        }
        catch (Throwable t) {
            log.error("Failed to generate authentication callbacks", t);
            throw new RemoteException("Failed to generate authentication callbacks", t);
        }
    }

    public void checkCompatibility(int minServerInterfaceVersion) throws RemoteException {
        if (minServerInterfaceVersion > 12) {
            throw new RemoteException("Incompatible server interface, a newer Ghidra Server version is required.");
        }
        if (minServerInterfaceVersion < 11) {
            throw new RemoteException("Incompatible server interface, the minimum supported Ghidra version is 9.0");
        }
    }

    public RemoteRepositoryServerHandle getRepositoryServer(Subject user, Callback[] authCallbacks) throws LoginException, RemoteException {
        NameCallback nameCb;
        System.gc();
        GhidraPrincipal principal = GhidraPrincipal.getGhidraPrincipal((Subject)user);
        if (principal == null) {
            throw new FailedLoginException("GhidraPrincipal required");
        }
        boolean anonymousAccess = false;
        String username = principal.getName();
        if (this.anonymousAuthModule != null && this.anonymousAuthModule.anonymousAccessRequested(authCallbacks)) {
            username = "-anonymous-";
            anonymousAccess = true;
            RepositoryManager.log(null, null, "Anonymous access allowed", principal.getName());
        } else if (this.authModule != null && (nameCb = AuthenticationModule.getFirstCallbackOfType(NameCallback.class, authCallbacks)) != null) {
            if (!this.authModule.isNameCallbackAllowed()) {
                RepositoryManager.log(null, null, "Illegal authentication callback: NameCallback not permitted", username);
                throw new LoginException("Illegal authentication callback");
            }
            String name = nameCb.getName();
            if (name == null) {
                RepositoryManager.log(null, null, "Illegal authentication callback: NameCallback must specify login name", username);
                throw new LoginException("Illegal authentication callback");
            }
            username = name;
        }
        RepositoryManager.log(null, null, "Repository server handle requested", username);
        boolean supportPasswordChange = false;
        if (!anonymousAccess) {
            if (this.sshAuthModule != null && this.sshAuthModule.hasSignedSSHCallback(authCallbacks)) {
                try {
                    username = this.sshAuthModule.authenticate(this.mgr.getUserManager(), user, authCallbacks);
                }
                catch (LoginException e) {
                    RepositoryManager.log(null, null, "SSH Authentication failed (" + e.getMessage() + ")", username);
                    throw e;
                }
            }
            if (this.authModule != null) {
                try {
                    username = this.authModule.authenticate(this.mgr.getUserManager(), user, authCallbacks);
                    anonymousAccess = "-anonymous-".equals(username);
                    if (!anonymousAccess) {
                        if (!this.mgr.getUserManager().isValidUser(username)) {
                            if (this.autoProvisionAuthedUsers) {
                                try {
                                    this.mgr.getUserManager().addUser(username);
                                    RepositoryManager.log(null, null, "User '" + username + "' successful auto provision", username);
                                }
                                catch (DuplicateNameException | IOException e) {
                                    RepositoryManager.log(null, null, "User '" + username + "' auto provision failed.  Cause: " + e.getMessage(), username);
                                    throw new LoginException("Error when trying to auto provision successfully authenticated user: " + username);
                                }
                            } else {
                                RepositoryManager.log(null, null, "User successfully authenticated, but does not exist in Ghidra user list: " + username, null);
                                throw new LoginException("Unknown user: " + username);
                            }
                        }
                        RepositoryManager.log(null, null, "User '" + username + "' authenticated", principal.getName());
                    }
                }
                catch (LoginException e) {
                    RepositoryManager.log(null, null, "Login failed (" + e.getMessage() + ")", username);
                    if (e instanceof FailedLoginException) {
                        throw new FailedLoginException("User authentication failed");
                    }
                    throw new LoginException("User login system failure");
                }
                if (this.authModule instanceof PasswordFileAuthenticationModule) {
                    supportPasswordChange = true;
                }
            } else if (!this.mgr.getUserManager().isValidUser(username)) {
                FailedLoginException e = new FailedLoginException("Unknown user: " + username);
                RepositoryManager.log(null, null, "Login failed (" + e.getMessage() + ")", username);
                throw e;
            }
        }
        if (anonymousAccess) {
            RepositoryManager.log(null, null, "Anonymous server access granted", null);
        }
        return new RepositoryServerHandleImpl(username, anonymousAccess, this.mgr, supportPasswordChange);
    }

    public void dispose() {
        try {
            GhidraServer.unexportObject(this, true);
        }
        catch (NoSuchObjectException noSuchObjectException) {
            // empty catch block
        }
        if (this.mgr != null) {
            this.mgr.dispose();
            this.mgr = null;
            log.info("Ghidra server terminated.");
        }
        if (server == this) {
            server = null;
        }
        if (this.blockStreamServer != null && this.blockStreamServer.isRunning()) {
            this.blockStreamServer.stopServer();
            this.blockStreamServer = null;
        }
    }

    private static void displayUsage(String msg) {
        if (msg != null) {
            System.out.println(msg);
        }
        System.out.println("Usage: java " + GhidraServer.class.getName() + USAGE_ARGS);
    }

    private static void displayHelp() {
        try (InputStream in = GhidraServer.class.getResourceAsStream(HELP_FILE);){
            List lines = FileUtilities.getLines((InputStream)in);
            lines.stream().forEach(s -> System.out.println((String)s));
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static InetAddress findHost() {
        for (int attempt = 0; attempt < 12; ++attempt) {
            try {
                if (attempt != 0) {
                    log.warn("Failed to discover IP interface - retry in 5 seconds...");
                    Thread.sleep(5000L);
                }
                Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
                while (e.hasMoreElements()) {
                    NetworkInterface nic = e.nextElement();
                    Enumeration<InetAddress> e2 = nic.getInetAddresses();
                    while (e2.hasMoreElements()) {
                        InetAddress ia = e2.nextElement();
                        if (ia.isLoopbackAddress()) continue;
                        if (attempt != 0) {
                            log.info("Discovered IP interface: " + ia.getHostAddress());
                        }
                        return ia;
                    }
                }
                log.info("No interfaces found: using loopback interface");
                return InetAddress.getLoopbackAddress();
            }
            catch (SocketException e) {
                continue;
            }
            catch (InterruptedException e) {
                break;
            }
        }
        return null;
    }

    private static String initRemoteAccessHostname() throws UnknownHostException {
        String hostname = System.getProperty(RMI_SERVER_PROPERTY);
        if (hostname == null) {
            if (bindAddress != null) {
                hostname = bindAddress.getHostAddress();
            } else {
                InetAddress localhost = InetAddress.getLocalHost();
                if (localhost.isLoopbackAddress() && (localhost = GhidraServer.findHost()) == null) {
                    log.fatal("Can't find host ip address!");
                    System.exit(-1);
                }
                hostname = localhost.getHostAddress();
            }
            System.setProperty(RMI_SERVER_PROPERTY, hostname);
        }
        return hostname;
    }

    private static File getServerCfgFile(String cfgFileName) {
        File tmp = new File(cfgFileName);
        if (tmp.isAbsolute()) {
            return tmp;
        }
        ResourceFile serverRoot = new ResourceFile(Application.getInstallationDirectory(), SystemUtilities.isInDevelopmentMode() ? "ghidra/Ghidra/RuntimeScripts/Common/server" : "server");
        if (serverRoot.getFile(false) == null) {
            System.err.println("Failed to resolve installation root directory!: " + serverRoot.getAbsolutePath());
            System.exit(-1);
        }
        return new File(serverRoot.getFile(false), cfgFileName);
    }

    public static synchronized void main(String[] args) {
        File serverRoot;
        if (serverSocketFactory != null) {
            throw new IllegalStateException("Server previously started within JVM");
        }
        if (args.length == 0) {
            GhidraServer.displayHelp();
            System.exit(-1);
        }
        if (server != null) {
            throw new AssertException("Server already started");
        }
        int basePort = 13100;
        AuthMode authMode = AuthMode.NO_AUTH_LOGIN;
        boolean nameCallbackAllowed = false;
        boolean altSSHLoginAllowed = false;
        boolean allowAnonymousAccess = false;
        String loginDomain = null;
        String rootPath = null;
        int defaultPasswordExpiration = -1;
        boolean autoProvision = false;
        File jaasConfigFile = null;
        InetNameLookup.setLookupEnabled(false);
        try {
            GhidraServerApplicationLayout layout = new GhidraServerApplicationLayout();
            ApplicationConfiguration configuration = new ApplicationConfiguration();
            configuration.setInitializeLogging(false);
            Application.initializeApplication((ApplicationLayout)layout, (ApplicationConfiguration)configuration);
        }
        catch (IOException e) {
            System.err.println("Failed to initialize the application!");
            System.exit(-1);
        }
        for (int i = 0; i < args.length; ++i) {
            String s = args[i];
            if (s.startsWith("-p")) {
                try {
                    basePort = Integer.parseInt(s.substring(2));
                }
                catch (NumberFormatException e1) {
                    basePort = -1;
                }
                if (basePort > 0 && basePort <= 65535) continue;
                GhidraServer.displayUsage("Invalid registry port specified");
                System.exit(-1);
                continue;
            }
            if (s.startsWith("-a") && s.length() == 3) {
                int authModeNum = Integer.MIN_VALUE;
                try {
                    authModeNum = Integer.parseInt(s.substring(2));
                }
                catch (NumberFormatException e1) {
                    GhidraServer.displayUsage("Invalid option: " + s);
                    System.exit(-1);
                }
                authMode = AuthMode.fromIndex(authModeNum);
                if (authMode != null) continue;
                GhidraServer.displayUsage("Invalid authentication mode: " + s);
                System.exit(-1);
                continue;
            }
            if (s.startsWith("-ip")) {
                int nextArgIndex = i + 1;
                String hostname = s.length() == 3 && nextArgIndex < args.length ? args[++i] : s.substring(3);
                if ((hostname = hostname.trim()).length() == 0 || hostname.startsWith("-")) {
                    GhidraServer.displayUsage("Missing -ip hostname");
                    System.exit(-1);
                }
                System.setProperty(RMI_SERVER_PROPERTY, hostname);
                continue;
            }
            if (s.startsWith("-i")) {
                int nextArgIndex = i + 1;
                String bindIp = s.length() == 2 && nextArgIndex < args.length ? args[++i] : s.substring(2);
                if ((bindIp = bindIp.trim()).length() == 0 || bindIp.startsWith("-")) {
                    GhidraServer.displayUsage("Missing -i interface bind address");
                    System.exit(-1);
                }
                try {
                    bindAddress = InetAddress.getByName(bindIp);
                }
                catch (UnknownHostException e) {
                    System.err.println("Unknown server interface bind address: " + bindIp);
                    System.exit(-1);
                }
                continue;
            }
            if (s.startsWith("-d") && s.length() > 2) {
                loginDomain = s.substring(2);
                continue;
            }
            if (s.equals("-u")) {
                nameCallbackAllowed = true;
                continue;
            }
            if (s.equals("-n")) {
                InetNameLookup.setLookupEnabled(true);
                continue;
            }
            if (s.equals("-anonymous")) {
                allowAnonymousAccess = true;
                continue;
            }
            if (s.equals("-ssh")) {
                altSSHLoginAllowed = true;
                continue;
            }
            if (s.startsWith("-e")) {
                try {
                    defaultPasswordExpiration = Integer.parseInt(s.substring(2));
                }
                catch (NumberFormatException nextArgIndex) {
                    // empty catch block
                }
                if (defaultPasswordExpiration < 0) {
                    GhidraServer.displayUsage("Invalid default password expiration");
                    System.exit(-1);
                    continue;
                }
                if (defaultPasswordExpiration != 0) continue;
                System.out.println("Default password expiration has been disbaled.");
                continue;
            }
            if (s.startsWith("-jaas")) {
                String jaasConfigFileStr = s.length() == 5 ? (++i < args.length ? args[i] : "") : s.substring(5);
                if ((jaasConfigFileStr = jaasConfigFileStr.trim()).isEmpty()) {
                    GhidraServer.displayUsage("Missing -jaas config file path argument");
                    System.exit(-1);
                }
                if ((jaasConfigFile = GhidraServer.getServerCfgFile(jaasConfigFileStr)).isFile()) continue;
                GhidraServer.displayUsage("JAAS config file (-jaas <configfile>) does not exist or is not file: " + jaasConfigFile.getAbsolutePath());
                System.exit(-1);
                continue;
            }
            if (s.equals("-autoProvision")) {
                autoProvision = true;
                continue;
            }
            if (i < args.length - 1) {
                GhidraServer.displayUsage("Invalid usage!");
                System.exit(-1);
            }
            rootPath = s;
        }
        if (rootPath == null) {
            GhidraServer.displayUsage("Repository directory must be specified!");
            System.exit(-1);
        }
        if (!(serverRoot = new File(rootPath)).isAbsolute()) {
            ResourceFile installRoot = Application.getInstallationDirectory();
            if (installRoot == null || installRoot.getFile(false) == null) {
                System.err.println("Failed to resolve installation root directory!");
                System.exit(-1);
            }
            serverRoot = new File(installRoot.getFile(false), rootPath);
        }
        if (authMode == AuthMode.JAAS_LOGIN && jaasConfigFile == null) {
            GhidraServer.displayUsage("JAAS config file argument (-jaas <configfile>) not specified");
            System.exit(-1);
        }
        try {
            serverRoot = serverRoot.getCanonicalFile();
        }
        catch (IOException e1) {
            System.err.println("Failed to resolve repository directory: " + serverRoot.getAbsolutePath());
            System.exit(-1);
        }
        if (!serverRoot.exists() && !serverRoot.mkdirs()) {
            System.err.println("Failed to create repository directory: " + serverRoot.getAbsolutePath());
            System.exit(-1);
        }
        File serverLogFile = new File(serverRoot, "server.log");
        Application.initializeLogging((File)serverLogFile, (File)serverLogFile);
        SSLContextInitializer.initialize();
        log = LogManager.getLogger(GhidraServer.class);
        ServerPortFactory.setBasePort(basePort);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            if (server != null) {
                server.dispose();
            }
        }, "Ghidra Server Disposer"));
        try {
            String hostname = GhidraServer.initRemoteAccessHostname();
            if (ApplicationKeyManagerFactory.getPreferredKeyStore() == null) {
                ApplicationKeyManagerFactory.setDefaultIdentity((X500Principal)new X500Principal("CN=GhidraServer"));
                ApplicationKeyManagerFactory.addSubjectAlternativeName((String)hostname);
            }
            if (!ApplicationKeyManagerFactory.initialize()) {
                log.fatal("Failed to initialize PKI/SSL keystore");
                System.exit(0);
                return;
            }
            log.info("Ghidra Server " + Application.getApplicationVersion());
            log.info("   Server remote access address: " + hostname);
            if (bindAddress == null) {
                log.info("   Server listening on all interfaces");
            } else {
                log.info("   Server listening on interface: " + bindAddress.getHostAddress());
            }
            log.info("   RMI Registry port: " + ServerPortFactory.getRMIRegistryPort());
            log.info("   RMI SSL port: " + ServerPortFactory.getRMISSLPort());
            log.info("   Block Stream port: " + ServerPortFactory.getStreamPort());
            log.info("   Block Stream compression: " + (RemoteBlockStreamHandle.enableCompressedSerializationOutput ? "enabled" : "disabled"));
            log.info("   Root: " + rootPath);
            log.info("   Auth: " + authMode.getDescription());
            if (authMode == AuthMode.PASSWORD_FILE_LOGIN && defaultPasswordExpiration >= 0) {
                log.info("   Default password expiration: " + (String)(defaultPasswordExpiration == 0 ? "disabled" : defaultPasswordExpiration + " days"));
            }
            if (authMode != AuthMode.PKI_LOGIN) {
                log.info("   Prompt for user ID: " + (nameCallbackAllowed ? "yes" : "no"));
            }
            if (altSSHLoginAllowed) {
                log.info("   SSH authentication option enabled");
            }
            log.info("   Anonymous server access: " + (allowAnonymousAccess ? "enabled" : "disabled"));
            String enabledCiphers = System.getProperty(TLS_ENABLED_CIPHERS_PROPERTY);
            if (enabledCiphers != null) {
                String[] cipherList = enabledCiphers.split(",");
                log.info("   Enabled cipher suites:");
                for (String s : cipherList) {
                    log.info("       " + s);
                }
            }
            serverSocketFactory = new SslRMIServerSocketFactory(null, GhidraServer.getEnabledTlsProtocols(), authMode == AuthMode.PKI_LOGIN){

                @Override
                public ServerSocket createServerSocket(int port) throws IOException {
                    return new GhidraSSLServerSocket(port, bindAddress, this.getEnabledCipherSuites(), this.getEnabledProtocols(), this.getNeedClientAuth());
                }
            };
            clientSocketFactory = new SslRMIClientSocketFactory();
            log.info(SystemUtilities.getUserName() + " starting Ghidra Server...");
            GhidraServer svr = new GhidraServer(serverRoot, authMode, loginDomain, nameCallbackAllowed, altSSHLoginAllowed, defaultPasswordExpiration, allowAnonymousAccess, autoProvision, jaasConfigFile);
            log.info("Registering Ghidra Server...");
            Registry registry = LocateRegistry.createRegistry(ServerPortFactory.getRMIRegistryPort(), clientSocketFactory, serverSocketFactory);
            registry.bind("GhidraServer9.0", svr);
            log.info("Registered Ghidra Server.");
        }
        catch (IOException e) {
            e.printStackTrace();
            log.error(e.getMessage());
            System.exit(-1);
        }
        catch (Throwable t) {
            log.fatal("Server error: " + t.getMessage(), t);
            System.exit(-1);
        }
    }

    private static String[] getEnabledTlsProtocols() {
        String protocolList = System.getProperty(TLS_SERVER_PROTOCOLS_PROPERTY);
        if (protocolList != null) {
            log.info("   Enabled protocols: " + protocolList);
            String[] protocols = protocolList.split(";");
            for (int i = 0; i < protocols.length; ++i) {
                protocols[i] = protocols[i].trim();
            }
            return protocols;
        }
        return null;
    }

    static synchronized void stop() {
        if (server == null) {
            throw new IllegalStateException("Invalid Stop request, Server is not running");
        }
        server.dispose();
    }

    public static RMIServerSocketFactory getRMIServerSocketFactory() {
        return serverSocketFactory;
    }

    public static RMIClientSocketFactory getRMIClientSocketFactory() {
        return clientSocketFactory;
    }

    private static void setGlobalSerializationFilter() throws IOException {
        String disabledStr = System.getProperty(SERIALIZATION_FILTER_DISABLED_PROPERTY);
        if (Boolean.valueOf(disabledStr).booleanValue()) {
            return;
        }
        final ObjectInputFilter patternFilter = GhidraServer.readSerialFilterPatternFile();
        ObjectInputFilter filter = new ObjectInputFilter(){

            @Override
            public ObjectInputFilter.Status checkInput(ObjectInputFilter.FilterInfo info) {
                Class<?> clazz = info.serialClass();
                ObjectInputFilter.Status status = patternFilter.checkInput(info);
                if (status != ObjectInputFilter.Status.UNDECIDED) {
                    if (status == ObjectInputFilter.Status.REJECTED) {
                        return this.serialReject(info, "failed by serial.filter pattern");
                    }
                    return status;
                }
                if (clazz == null) {
                    return ObjectInputFilter.Status.ALLOWED;
                }
                Class<?> componentType = clazz.getComponentType();
                if (componentType != null && componentType.isPrimitive()) {
                    return ObjectInputFilter.Status.ALLOWED;
                }
                return this.serialReject(info, "not allowed");
            }

            private ObjectInputFilter.Status serialReject(ObjectInputFilter.FilterInfo info, String reason) {
                String clientHost = RepositoryManager.getRMIClient();
                StringBuilder buf = new StringBuilder();
                buf.append("Rejected class serialization");
                if (clientHost != null) {
                    buf.append(" from ");
                    buf.append(clientHost);
                }
                buf.append("(");
                buf.append(reason);
                buf.append(")");
                Class<?> serialClass = info.serialClass();
                if (serialClass != null) {
                    buf.append(": ");
                    buf.append(serialClass.getCanonicalName());
                    buf.append(" ");
                    if (serialClass.getComponentType() != null) {
                        buf.append("(");
                        buf.append("array-length=");
                        buf.append(info.arrayLength());
                        buf.append(")");
                    }
                }
                log.error(buf.toString());
                return ObjectInputFilter.Status.REJECTED;
            }
        };
        ObjectInputFilter.Config.setSerialFilter(filter);
    }

    private static ObjectInputFilter readSerialFilterPatternFile() throws IOException {
        File serialFilterFile = Application.getModuleDataFile((String)SERIAL_FILTER_FILE).getFile(false);
        if (serialFilterFile == null) {
            throw new FileNotFoundException("serial.filter not found");
        }
        try {
            StringBuilder buf = new StringBuilder();
            try (FileReader fr = new FileReader(serialFilterFile);
                 BufferedReader r = new BufferedReader(fr);){
                String line = r.readLine();
                while (line != null) {
                    int ix = line.indexOf(35);
                    if (ix >= 0) {
                        line = line.substring(0, ix);
                    }
                    if ((line = line.trim()).length() != 0) {
                        if (!line.endsWith(";")) {
                            throw new IllegalArgumentException("all filter statements must end with `;`");
                        }
                        if (line.length() != 0) {
                            buf.append(line);
                        }
                    }
                    line = r.readLine();
                }
            }
            return ObjectInputFilter.Config.createFilter(buf.toString());
        }
        catch (Exception e) {
            throw new IOException("Failed to parse serial.filter", e);
        }
    }

    static {
        HELP_FILE = "ServerHelp.txt";
        USAGE_ARGS = "[-ip <hostname>] [-i #.#.#.#] [-p#] [-n] [-a#] [-d<ad_domain>] [-e<days>] [-jaas <config_file>] [-u] [-autoProvision] [-anonymous] [-ssh] <repository_path>";
    }

    static enum AuthMode {
        NO_AUTH_LOGIN("None"),
        PASSWORD_FILE_LOGIN("Password File"),
        KRB5_AD_LOGIN("Active Directory via Kerberos"),
        PKI_LOGIN("PKI"),
        JAAS_LOGIN("JAAS");

        private String description;

        private AuthMode(String description) {
            this.description = description;
        }

        public String getDescription() {
            return this.description;
        }

        public static AuthMode fromIndex(int index) {
            switch (index) {
                case 0: {
                    return PASSWORD_FILE_LOGIN;
                }
                case 1: {
                    return KRB5_AD_LOGIN;
                }
                case 2: {
                    return PKI_LOGIN;
                }
                case 4: {
                    return JAAS_LOGIN;
                }
            }
            return null;
        }
    }
}

