/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.bin.format.dwarf;

import ghidra.app.plugin.core.datamgr.util.DataTypeUtils;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.format.dwarf.DWARFCompilationUnit;
import ghidra.app.util.bin.format.dwarf.DWARFDataTypeManager;
import ghidra.app.util.bin.format.dwarf.DWARFException;
import ghidra.app.util.bin.format.dwarf.DWARFFunctionImporter;
import ghidra.app.util.bin.format.dwarf.DWARFImportOptions;
import ghidra.app.util.bin.format.dwarf.DWARFImportSummary;
import ghidra.app.util.bin.format.dwarf.DWARFProgram;
import ghidra.app.util.bin.format.dwarf.DWARFSourceInfo;
import ghidra.app.util.bin.format.dwarf.line.DWARFLine;
import ghidra.framework.store.LockException;
import ghidra.program.database.sourcemap.SourceFile;
import ghidra.program.database.sourcemap.SourceFileIdType;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.data.Array;
import ghidra.program.model.data.Category;
import ghidra.program.model.data.CategoryPath;
import ghidra.program.model.data.Composite;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeComponent;
import ghidra.program.model.data.DataTypePath;
import ghidra.program.model.data.Pointer;
import ghidra.program.model.data.ProgramBasedDataTypeManager;
import ghidra.program.model.listing.Program;
import ghidra.program.model.sourcemap.SourceFileManager;
import ghidra.util.Msg;
import ghidra.util.SourceFileUtils;
import ghidra.util.Swing;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.commons.io.FilenameUtils;
import utility.function.Dummy;

public class DWARFImporter {
    private static final Set<String> SOURCEFILENAMES_IGNORE = Set.of("<autogenerated>");
    private DWARFProgram prog;
    private DWARFDataTypeManager dwarfDTM;
    private TaskMonitor monitor;
    private static final int MAX_NUM_SOURCE_LINE_ERROR_REPORTS = 200;
    private static final int MAX_NUM_SOURCE_LINE_WARNING_REPORTS = 200;
    private int numSourceLineErrorReports = 0;
    private int numSourceLineWarningReports = 0;
    public static final String DEFAULT_COMPILATION_DIR = "DWARF_DEFAULT_COMP_DIR";

    public DWARFImporter(DWARFProgram prog, TaskMonitor monitor) {
        this.prog = prog;
        this.monitor = monitor;
        this.dwarfDTM = prog.getDwarfDTM();
    }

    private void moveTypesIntoSourceFolders() throws CancelledException {
        List<DataTypePath> importedTypes = this.dwarfDTM.getImportedTypes();
        Collections.sort(importedTypes, (dtp1, dtp2) -> dtp1.getCategoryPath().getPath().compareTo(dtp2.getCategoryPath().getPath()));
        this.monitor.setIndeterminate(false);
        this.monitor.setShowProgressValue(true);
        this.monitor.initialize((long)importedTypes.size());
        this.monitor.setMessage("DWARF Move Types");
        CategoryPath unCatRootCp = this.prog.getUncategorizedRootDNI().getOrganizationalCategoryPath();
        CategoryPath rootCP = this.prog.getRootDNI().asCategoryPath();
        for (DataTypePath dataTypePath : importedTypes) {
            DWARFSourceInfo dsi;
            DataType dataType;
            this.monitor.checkCancelled();
            this.monitor.incrementProgress(1L);
            if (this.monitor.getProgress() % 5L == 0L) {
                Swing.runNow((Runnable)Dummy.runnable());
            }
            if ((dataType = this.prog.getGhidraProgram().getDataTypeManager().getDataType(dataTypePath)) == null || dataType instanceof Pointer || dataType instanceof Array || (dsi = this.dwarfDTM.getSourceInfo(dataType)) == null || dsi.filename() == null) continue;
            CategoryPath dataTypeOrigCP = dataType.getCategoryPath();
            CategoryPath newRoot = new CategoryPath(rootCP, new String[]{dsi.filename()});
            CategoryPath newCP = this.rehomeCategoryPathSubTree(unCatRootCp, newRoot, dataTypeOrigCP);
            if (newCP == null) continue;
            try {
                dataType.setCategoryPath(newCP);
                if (dataType instanceof Composite) {
                    this.fixupAnonStructMembers((Composite)dataType, dataTypeOrigCP, newCP);
                }
                this.deleteEmptyCategoryPaths(dataTypeOrigCP);
            }
            catch (DuplicateNameException e) {
                Msg.error((Object)this, (Object)("Failed to move " + String.valueOf(dataType.getDataTypePath()) + " to " + String.valueOf(newCP)));
            }
        }
        this.monitor.setMessage("DWARF Move Types - Done");
    }

    private void fixupAnonStructMembers(Composite compositeDataType, CategoryPath origCategoryPath, CategoryPath newCP) throws DuplicateNameException {
        CategoryPath origCompositeNSCP = new CategoryPath(origCategoryPath, new String[]{compositeDataType.getName()});
        CategoryPath destCompositeNSCP = new CategoryPath(newCP, new String[]{compositeDataType.getName()});
        for (DataTypeComponent component : compositeDataType.getDefinedComponents()) {
            DataType dtcDT = component.getDataType();
            if (dtcDT instanceof Array || dtcDT instanceof Pointer) {
                dtcDT = DataTypeUtils.getNamedBaseDataType(dtcDT);
            }
            if (!dtcDT.getCategoryPath().equals((Object)origCompositeNSCP) || this.dwarfDTM.getSourceInfo(dtcDT) != null) continue;
            dtcDT.setCategoryPath(destCompositeNSCP);
        }
        this.deleteEmptyCategoryPaths(origCompositeNSCP);
    }

    private void deleteEmptyCategoryPaths(CategoryPath cp) {
        ProgramBasedDataTypeManager dtm = this.prog.getGhidraProgram().getDataTypeManager();
        while (!CategoryPath.ROOT.equals((Object)cp)) {
            Category cat = dtm.getCategory(cp);
            Category parentCat = dtm.getCategory(cp.getParent());
            if (cat == null || parentCat == null || cat.getDataTypes().length != 0 || cat.getCategories().length != 0) break;
            if (!parentCat.removeEmptyCategory(cat.getName(), this.monitor)) {
                Msg.error((Object)this, (Object)("Failed to delete empty category " + String.valueOf(cp)));
                break;
            }
            cp = parentCat.getCategoryPath();
        }
    }

    private CategoryPath rehomeCategoryPathSubTree(CategoryPath origRoot, CategoryPath newRoot, CategoryPath cp) {
        if (origRoot.equals((Object)cp)) {
            return newRoot;
        }
        List origRootParts = origRoot.asList();
        List cpParts = cp.asList();
        if (cpParts.size() < origRootParts.size() || !origRootParts.equals(cpParts.subList(0, origRootParts.size()))) {
            return null;
        }
        return new CategoryPath(newRoot, cpParts.subList(origRootParts.size(), cpParts.size()));
    }

    private void addSourceLineInfo(BinaryReader reader) throws CancelledException, IOException, LockException {
        if (reader == null) {
            Msg.warn((Object)this, (Object)"Can't add source line info - reader is null");
            return;
        }
        int entryCount = 0;
        Program ghidraProgram = this.prog.getGhidraProgram();
        long maxLength = this.prog.getImportOptions().getMaxSourceMapEntryLength();
        List<DWARFCompilationUnit> compUnits = this.prog.getCompilationUnits();
        this.monitor.initialize((long)compUnits.size(), "DWARF: Reading Source Information");
        SourceFileManager sourceManager = ghidraProgram.getSourceFileManager();
        HashMap<DWARFLine.SourceFileInfo, SourceFile> sourceFileInfoToSourceFile = new HashMap<DWARFLine.SourceFileInfo, SourceFile>();
        HashSet<DWARFLine.SourceFileInfo> badSourceFileInfo = new HashSet<DWARFLine.SourceFileInfo>();
        ArrayList<DWARFLine.SourceFileAddr> sourceInfo = new ArrayList<DWARFLine.SourceFileAddr>();
        for (DWARFCompilationUnit cu : compUnits) {
            DWARFLine dLine = cu.getLine();
            this.monitor.increment(1L);
            for (DWARFLine.SourceFileInfo sfi : dLine.getAllSourceFileInfos()) {
                if (sourceFileInfoToSourceFile.containsKey(sfi) || badSourceFileInfo.contains(sfi)) continue;
                try {
                    String path = SourceFileUtils.normalizeDwarfPath(sfi.filePath(), DEFAULT_COMPILATION_DIR);
                    SourceFileIdType type = sfi.md5() == null ? SourceFileIdType.NONE : SourceFileIdType.MD5;
                    SourceFile sFile = new SourceFile(path, type, sfi.md5());
                    sourceManager.addSourceFile(sFile);
                    sourceFileInfoToSourceFile.put(sfi, sFile);
                }
                catch (IllegalArgumentException e) {
                    String errorString = "Exception creating source file: " + e.getMessage();
                    if (this.numSourceLineErrorReports++ < 200) {
                        Msg.error((Object)this, (Object)errorString);
                    }
                    badSourceFileInfo.add(sfi);
                }
            }
            sourceInfo.addAll(cu.getLine().getAllSourceFileAddrInfo(cu, reader));
        }
        this.monitor.setIndeterminate(true);
        this.monitor.setMessage("Sorting " + sourceInfo.size() + " entries");
        sourceInfo.sort((i, j) -> Long.compareUnsigned(i.address(), j.address()));
        this.monitor.setIndeterminate(false);
        this.monitor.initialize((long)sourceInfo.size(), "DWARF: Applying Source Map Info");
        AddressSet warnedAddresses = new AddressSet();
        for (int i2 = 0; i2 < sourceInfo.size(); ++i2) {
            Object warningString;
            Address addr;
            DWARFLine.SourceFileInfo sfi;
            this.monitor.increment(1L);
            DWARFLine.SourceFileAddr sfa = (DWARFLine.SourceFileAddr)sourceInfo.get(i2);
            if (SOURCEFILENAMES_IGNORE.contains(sfa.fileName()) || SOURCEFILENAMES_IGNORE.contains(FilenameUtils.getName((String)sfa.fileName())) || sfa.isEndSequence() || sfa.fileName() == null || badSourceFileInfo.contains(sfi = new DWARFLine.SourceFileInfo(sfa.fileName(), sfa.md5())) || warnedAddresses.contains(addr = this.prog.getCodeAddress(sfa.address()))) continue;
            if (!ghidraProgram.getMemory().getExecuteSet().contains(addr)) {
                String warningString2 = "entry for non-executable address; skipping: file %s line %d address: %s %x".formatted(sfa.fileName(), sfa.lineNum(), addr.toString(), sfa.address());
                if (this.numSourceLineWarningReports++ < 200) {
                    this.prog.logWarningAt(addr, addr.toString(), warningString2);
                }
                warnedAddresses.add(addr);
                continue;
            }
            long length = this.getLength(i2, sourceInfo);
            if (length < 0L) {
                length = 0L;
                warningString = "Error calculating entry length for file %s line %d address %s %x; replacing " + "with length 0 entry".formatted(sfa.fileName(), sfa.lineNum(), addr.toString(), sfa.address());
                if (this.numSourceLineWarningReports++ < 200) {
                    this.prog.logWarningAt(addr, addr.toString(), (String)warningString);
                }
            }
            if (length > maxLength) {
                warningString = "entry for file %s line %d address: %s %x length %d too large, replacing with length 0 entry".formatted(sfa.fileName(), sfa.lineNum(), addr.toString(), sfa.address(), length);
                length = 0L;
                if (this.numSourceLineWarningReports++ < 200) {
                    this.prog.logWarningAt(addr, addr.toString(), (String)warningString);
                }
            }
            SourceFile source = (SourceFile)sourceFileInfoToSourceFile.get(sfi);
            try {
                sourceManager.addSourceMapEntry(source, sfa.lineNum(), addr, length);
            }
            catch (AddressOverflowException | IllegalArgumentException e) {
                String errorString = e.getClass().getName() + " for source map entry %s %d %s %x %d".formatted(source.getFilename(), sfa.lineNum(), addr.toString(), sfa.address(), length);
                if (this.numSourceLineErrorReports++ >= 200) continue;
                this.reportError(errorString, addr);
                continue;
            }
            ++entryCount;
        }
        if (this.numSourceLineWarningReports >= 200) {
            Msg.warn((Object)this, (Object)"Additional warnings suppressed (%d total warnings)".formatted(this.numSourceLineWarningReports));
        }
        if (this.numSourceLineErrorReports >= 200) {
            Msg.error((Object)this, (Object)"Additional errors suppressed (%d total errors)".formatted(this.numSourceLineErrorReports));
        }
        Msg.info((Object)this, (Object)"Added %d source map entries".formatted(entryCount));
    }

    private void reportError(String errorString, Address addr) {
        if (this.prog.getImportOptions().isUseBookmarks()) {
            this.prog.getGhidraProgram().getBookmarkManager().setBookmark(addr, "Error", "DWARF", errorString);
        } else {
            Msg.error((Object)this, (Object)errorString);
        }
    }

    private long getLength(int i, List<DWARFLine.SourceFileAddr> allSFA) {
        DWARFLine.SourceFileAddr iAddr = allSFA.get(i);
        long iOffset = iAddr.address();
        for (int j = i + 1; j < allSFA.size(); ++j) {
            DWARFLine.SourceFileAddr current = allSFA.get(j);
            long currentAddr = current.address();
            if (current.isEndSequence()) {
                return currentAddr + 1L - iOffset;
            }
            if (currentAddr == iOffset) continue;
            return currentAddr - iOffset;
        }
        return -1L;
    }

    public DWARFImportSummary performImport() throws IOException, DWARFException, CancelledException {
        this.monitor.setIndeterminate(false);
        this.monitor.setShowProgressValue(true);
        DWARFImportOptions importOptions = this.prog.getImportOptions();
        DWARFImportSummary importSummary = this.prog.getImportSummary();
        long start_ts = System.currentTimeMillis();
        if (importOptions.isImportDataTypes()) {
            this.dwarfDTM.importAllDataTypes(this.monitor);
            this.prog.getGhidraProgram().flushEvents();
            importSummary.dataTypeElapsedMS = System.currentTimeMillis() - start_ts;
        }
        if (importOptions.isImportFuncs()) {
            long funcstart_ts = System.currentTimeMillis();
            DWARFFunctionImporter dfi = new DWARFFunctionImporter(this.prog, this.monitor);
            dfi.importFunctions();
            importSummary.funcsElapsedMS = System.currentTimeMillis() - funcstart_ts;
        }
        if (importOptions.isOrganizeTypesBySourceFile()) {
            this.moveTypesIntoSourceFolders();
        }
        if (importOptions.isOutputSourceLineInfo()) {
            if (!this.prog.getGhidraProgram().hasExclusiveAccess()) {
                Msg.showError((Object)this, null, (String)"Unable to add source map info", (Object)"Exclusive access to the program is required to add source map info");
            } else {
                try {
                    this.addSourceLineInfo(this.prog.getDebugLineBR());
                }
                catch (LockException e) {
                    throw new AssertException("LockException after exclusive access verified");
                }
            }
        }
        importSummary.totalElapsedMS = System.currentTimeMillis() - start_ts;
        return importSummary;
    }
}

