/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.analysis;

import generic.jar.ResourceFile;
import ghidra.app.cmd.comments.SetCommentCmd;
import ghidra.app.plugin.core.analysis.AutoAnalysisManager;
import ghidra.app.plugin.core.analysis.ConstantPropagationContextEvaluator;
import ghidra.app.plugin.core.analysis.NonReturningFunctionNames;
import ghidra.app.services.AbstractAnalyzer;
import ghidra.app.services.AnalysisPriority;
import ghidra.app.services.AnalyzerType;
import ghidra.app.util.MemoryBlockUtils;
import ghidra.app.util.bin.format.golang.GoConstants;
import ghidra.app.util.bin.format.golang.GoFunctionFixup;
import ghidra.app.util.bin.format.golang.GoParamStorageAllocator;
import ghidra.app.util.bin.format.golang.GoRegisterInfo;
import ghidra.app.util.bin.format.golang.rtti.GoFuncData;
import ghidra.app.util.bin.format.golang.rtti.GoModuledata;
import ghidra.app.util.bin.format.golang.rtti.GoRttiMapper;
import ghidra.app.util.bin.format.golang.rtti.GoSourceFileInfo;
import ghidra.app.util.bin.format.golang.rtti.GoSymbolName;
import ghidra.app.util.bin.format.golang.rtti.GoSymbolNameType;
import ghidra.app.util.bin.format.golang.rtti.GoTypeManager;
import ghidra.app.util.bin.format.golang.rtti.types.GoBaseType;
import ghidra.app.util.bin.format.golang.rtti.types.GoType;
import ghidra.app.util.bin.format.golang.rtti.types.GoTypeBridge;
import ghidra.app.util.bin.format.golang.structmapping.MarkupSession;
import ghidra.app.util.importer.MessageLog;
import ghidra.app.util.viewer.field.AddressAnnotatedStringHandler;
import ghidra.docking.settings.Settings;
import ghidra.framework.cmd.BackgroundCommand;
import ghidra.framework.options.Options;
import ghidra.framework.store.LockException;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressRange;
import ghidra.program.model.address.AddressSet;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.data.AbstractIntegerDataType;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.DataTypeManager;
import ghidra.program.model.data.DataUtilities;
import ghidra.program.model.data.FunctionDefinitionDataType;
import ghidra.program.model.data.MutabilitySettingsDefinition;
import ghidra.program.model.data.Pointer;
import ghidra.program.model.data.Structure;
import ghidra.program.model.data.TypeDef;
import ghidra.program.model.data.VoidDataType;
import ghidra.program.model.lang.Language;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.CircularDependencyException;
import ghidra.program.model.listing.CommentType;
import ghidra.program.model.listing.ContextChangeException;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Function;
import ghidra.program.model.listing.FunctionIterator;
import ghidra.program.model.listing.FunctionManager;
import ghidra.program.model.listing.FunctionSignature;
import ghidra.program.model.listing.Parameter;
import ghidra.program.model.listing.ParameterImpl;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ReturnParameterImpl;
import ghidra.program.model.listing.Variable;
import ghidra.program.model.listing.VariableStorage;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.program.model.mem.MemoryBlockException;
import ghidra.program.model.pcode.HighFunctionDBUtil;
import ghidra.program.model.symbol.Namespace;
import ghidra.program.model.symbol.Reference;
import ghidra.program.model.symbol.ReferenceManager;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolUtilities;
import ghidra.program.model.util.CodeUnitInsertionException;
import ghidra.program.util.ContextEvaluator;
import ghidra.program.util.FunctionUtility;
import ghidra.program.util.SymbolicPropogator;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.exception.NotFoundException;
import ghidra.util.task.TaskMonitor;
import ghidra.util.task.UnknownProgressWrappingTaskMonitor;
import ghidra.xml.XmlParseException;
import java.io.IOException;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import utilities.util.FileUtilities;

public class GolangSymbolAnalyzer
extends AbstractAnalyzer {
    private static final AnalysisPriority GOLANG_ANALYSIS_PRIORITY = AnalysisPriority.FORMAT_ANALYSIS.after().after();
    private static final AnalysisPriority PROP_RTTI_PRIORITY = AnalysisPriority.REFERENCE_ANALYSIS.after();
    private static final AnalysisPriority FIX_CLOSURES_PRIORITY = PROP_RTTI_PRIORITY.after();
    static final AnalysisPriority STRINGS_PRIORITY = FIX_CLOSURES_PRIORITY.after();
    private static final AnalysisPriority FIX_GCWRITEBARRIER_PRIORITY = STRINGS_PRIORITY.after();
    private static final String NAME = "Golang Symbols";
    private static final String DESCRIPTION = "Analyze Go binaries for RTTI and function symbols.\n'Apply Data Archives' and 'Shared Return Calls' analyzers should be disabled for best results.";
    private static final String ANALYZED_FLAG_OPTION_NAME = "Golang Analyzed";
    private GolangAnalyzerOptions analyzerOptions = new GolangAnalyzerOptions();
    private GoRttiMapper goBinary;
    private GoTypeManager goTypes;
    private Program program;
    private MarkupSession markupSession;
    private AutoAnalysisManager aam;
    private long lastTxId = -1L;
    private static final Set<String> FUNCNAMES_TO_IGNORE = Set.of("go:buildid", "go.buildid");

    public GolangSymbolAnalyzer() {
        super(NAME, DESCRIPTION, AnalyzerType.BYTE_ANALYZER);
        this.setPriority(GOLANG_ANALYSIS_PRIORITY);
        this.setDefaultEnablement(true);
    }

    @Override
    public boolean canAnalyze(Program program) {
        return GoRttiMapper.isGolangProgram(program);
    }

    @Override
    public boolean added(Program program, AddressSetView set, TaskMonitor monitor, MessageLog log) throws CancelledException {
        long txId = program.getCurrentTransactionInfo().getID();
        if (txId == this.lastTxId) {
            return true;
        }
        this.lastTxId = txId;
        if (GolangSymbolAnalyzer.isAlreadyAnalyzed(program)) {
            Msg.info((Object)this, (Object)"Go analysis already performed, skipping.");
            return false;
        }
        monitor.setMessage("Go symbol analyzer");
        this.program = program;
        this.aam = AutoAnalysisManager.getAnalysisManager(program);
        this.goBinary = GoRttiMapper.getSharedGoBinary(program, monitor);
        if (this.goBinary == null) {
            Msg.error((Object)this, (Object)"Go symbol analyzer error: unable to get GoRttiMapper");
            return false;
        }
        this.goTypes = this.goBinary.getGoTypes();
        this.markupSession = this.goBinary.createMarkupSession(monitor);
        try {
            this.goTypes.recoverGhidraDataTypes(monitor);
            this.goTypes.markupGoTypes(this.markupSession, monitor);
            GoModuledata firstModule = this.goBinary.getFirstModule();
            if (firstModule != null) {
                this.markupSession.labelStructure(firstModule, "firstmoduledata", null);
                this.markupSession.markup(firstModule, false);
            }
            this.markupWellknownSymbols();
            this.setupProgramContext();
            this.markupGoFunctions(monitor);
            if (this.analyzerOptions.fixupDuffFunctions) {
                this.fixDuffFunctions();
            }
            if (this.analyzerOptions.fixupGcWriteBarrierFunctions) {
                this.fixGcWriteBarrierFunctions();
            }
            if (this.analyzerOptions.propagateRtti) {
                Msg.info((Object)this, (Object)"Go symbol analyzer: scheduling RTTI propagation after reference analysis");
                this.aam.schedule(new PropagateRttiBackgroundCommand(this.goBinary, this.markupSession), PROP_RTTI_PRIORITY.priority());
                Msg.info((Object)this, (Object)"Go symbol analyzer: scheduling closure function fixup");
                this.aam.schedule(new FixClosureFuncArgsBackgroundCommand(this.goBinary), FIX_CLOSURES_PRIORITY.priority());
            }
            if (this.analyzerOptions.fixupGcWriteBarrierFlag) {
                Msg.info((Object)this, (Object)"Go symbol analyzer: scheduling gcWriteBarrier flag fixup");
                this.aam.schedule(new FixGcWriteBarrierFlagBackgroundCommand(this.goBinary, this.markupSession), FIX_GCWRITEBARRIER_PRIORITY.priority());
            }
            program.getOptions("Program Information").setBoolean(ANALYZED_FLAG_OPTION_NAME, true);
            return true;
        }
        catch (IOException e) {
            Msg.error((Object)this, (Object)"Go analysis failure", (Throwable)e);
            return false;
        }
    }

    @Override
    public void registerOptions(Options options, Program program) {
        this.analyzerOptions.registerOptions(options, program);
    }

    @Override
    public void optionsChanged(Options options, Program program) {
        this.analyzerOptions.optionsChanged(options, program);
    }

    private void markupWellknownSymbols() throws IOException {
        Symbol g0 = this.goBinary.getGoSymbol("runtime.g0");
        Structure gStruct = this.goTypes.findDataType("runtime.g", Structure.class);
        if (g0 != null && gStruct != null) {
            this.markupSession.markupAddressIfUndefined(g0.getAddress(), (DataType)gStruct);
        }
        Symbol m0 = this.goBinary.getGoSymbol("runtime.m0");
        Structure mStruct = this.goTypes.findDataType("runtime.m", Structure.class);
        if (m0 != null && mStruct != null) {
            this.markupSession.markupAddressIfUndefined(m0.getAddress(), (DataType)mStruct);
        }
    }

    private void markupGoFunctions(TaskMonitor monitor) throws IOException, CancelledException {
        Set<String> noreturnFuncNames = this.readNoReturnFuncNames();
        int noreturnFuncCount = 0;
        int functionSignatureFromBootstrap = 0;
        int functionSignatureFromMethod = 0;
        List<GoFuncData> funcs = this.goBinary.getAllFunctions();
        monitor.initialize((long)funcs.size(), "Fixing Go function signatures");
        for (GoFuncData funcdata : funcs) {
            GoRttiMapper.FuncDefResult funcDefResult;
            Function func;
            monitor.increment();
            Address funcAddr = funcdata.getFuncAddress();
            GoSymbolName funcSymbolNameInfo = funcdata.getSymbolName();
            String funcname = SymbolUtilities.replaceInvalidChars((String)funcSymbolNameInfo.asString(), (boolean)true);
            Namespace funcns = funcSymbolNameInfo.getSymbolNamespace(this.program);
            if (FUNCNAMES_TO_IGNORE.contains(funcSymbolNameInfo.asString()) || (func = this.markupSession.createFunctionIfMissing(funcname, funcns, funcAddr)) == null || func.getSignatureSource().isHigherPriorityThan(SourceType.IMPORTED)) continue;
            boolean prevNoReturnFlag = func.hasNoReturn();
            this.markupSession.appendComment(func, "Golang function info: ", AddressAnnotatedStringHandler.createAddressAnnotationString(funcdata.getStructureContext().getStructureAddress(), "Flags: %s".formatted(funcdata.getFlags())));
            if (!funcSymbolNameInfo.asString().equals(funcname)) {
                this.markupSession.appendComment(func, "Golang original name: ", funcSymbolNameInfo.asString());
            }
            GoSourceFileInfo sfi = null;
            if (this.analyzerOptions.outputSourceInfo && (sfi = funcdata.getSourceFileInfo()) != null) {
                this.markupSession.appendComment(func, "Golang source: ", sfi.getDescription());
                funcdata.markupSourceFileInfo();
            }
            if (funcdata.getFlags().isEmpty()) {
                this.markupSession.appendComment(func, null, "Golang stacktrace signature: " + funcdata.recoverFunctionSignature());
            }
            if ((funcDefResult = this.goBinary.getFuncDefFor(funcdata)) != null) {
                Set<GoRttiMapper.FuncDefFlags> flags = funcDefResult.flags();
                String flagStr = !flags.isEmpty() ? " " + flags.toString().toLowerCase() : "";
                String snapshotStr = funcDefResult.funcDefStr();
                if (!flagStr.isEmpty() || !snapshotStr.isEmpty()) {
                    this.markupSession.appendComment(func, null, "Golang signature%s: %s".formatted(flagStr, snapshotStr));
                }
                if (flags.contains((Object)GoRttiMapper.FuncDefFlags.FROM_SNAPSHOT)) {
                    ++functionSignatureFromBootstrap;
                }
                if (flags.contains((Object)GoRttiMapper.FuncDefFlags.FROM_RTTI_METHOD)) {
                    ++functionSignatureFromMethod;
                }
                GoFunctionFixup ff = new GoFunctionFixup(func, (FunctionSignature)funcDefResult.funcDef(), this.goBinary.getCallingConventionFor(funcdata), this.goBinary.newStorageAllocator());
                try {
                    ff.apply();
                }
                catch (DuplicateNameException | InvalidInputException | IllegalArgumentException e) {
                    MarkupSession.logWarningAt(this.program, func.getEntryPoint(), "Failed to update function signature: " + e.getMessage());
                    continue;
                }
                if (funcDefResult.symbolName().hasReceiver()) {
                    GoType recvType = funcDefResult.recvType();
                    Address typeStructAddr = recvType != null && !(recvType instanceof GoTypeBridge) ? recvType.getStructureContext().getStructureAddress() : null;
                    String typeStr = typeStructAddr != null ? AddressAnnotatedStringHandler.createAddressAnnotationString(typeStructAddr, recvType.getName()) : funcDefResult.symbolName().receiverString();
                    this.markupSession.appendComment(func, "", "Golang method in type %s".formatted(typeStr));
                }
            }
            if (noreturnFuncNames.contains(funcname) && !func.hasNoReturn()) {
                func.setNoReturn(true);
            }
            if (!func.hasNoReturn() || func.hasNoReturn() == prevNoReturnFlag) continue;
            ++noreturnFuncCount;
        }
        Msg.info((Object)this, (Object)"Marked %d Go funcs as NoReturn".formatted(noreturnFuncCount));
        Msg.info((Object)this, (Object)"Fixed %d Go function signatures from runtime snapshot signatures".formatted(functionSignatureFromBootstrap));
        Msg.info((Object)this, (Object)"Fixed %d Go function signatures from method info".formatted(functionSignatureFromMethod));
    }

    private void fixGcWriteBarrierFunctions() {
        if (GoConstants.GCWRITE_BUFFERED_VERS.contains(this.goBinary.getGoVer())) {
            this.fixGcWriteBarrierBufferedFunctions();
        } else if (GoConstants.GCWRITE_BATCH_VERS.contains(this.goBinary.getGoVer())) {
            this.fixGcWriteBarrierBatchFunctions();
        }
    }

    private void fixGcWriteBarrierBatchFunctions() {
        String ccname = "gcwrite_batch";
        if (!this.goBinary.hasCallingConvention(ccname)) {
            Msg.warn((Object)this, (Object)("Missing " + ccname + " from this arch's .cspec"));
            return;
        }
        try {
            Function func;
            ReturnParameterImpl retVal = new ReturnParameterImpl((DataType)this.goTypes.getDTM().getPointer(null), this.program);
            GoFuncData funcData = this.goBinary.getFunctionByName("gcWriteBarrier");
            Function function = func = funcData != null ? funcData.getFunction() : null;
            if (func != null) {
                List<ParameterImpl> params = List.of(new ParameterImpl("numbytes", this.goTypes.findDataType("uint"), this.program, SourceType.ANALYSIS));
                func.updateFunction(ccname, (Variable)retVal, params, Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
            }
            for (int i = 1; i <= 8; ++i) {
                funcData = this.goBinary.getFunctionByName("runtime.gcWriteBarrier" + i);
                Function function2 = func = funcData != null ? funcData.getFunction() : null;
                if (func == null) continue;
                func.updateFunction(ccname, (Variable)retVal, List.of(), Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
            }
        }
        catch (DuplicateNameException | InvalidInputException | IOException e) {
            Msg.error((Object)this, (Object)"Failed to update gcwrite function", (Throwable)e);
        }
    }

    private void fixGcWriteBarrierBufferedFunctions() {
        String ccname = "gcwrite_buffered";
        if (!this.goBinary.hasCallingConvention(ccname)) {
            Msg.warn((Object)this, (Object)("Missing " + ccname + " from this arch's .cspec"));
            return;
        }
        try {
            Function func;
            Pointer voidPtr = this.goTypes.getDTM().getPointer(null);
            ReturnParameterImpl retVal = new ReturnParameterImpl((DataType)VoidDataType.dataType, this.program);
            List<ParameterImpl> params = List.of(new ParameterImpl("value", (DataType)voidPtr, this.program, SourceType.ANALYSIS), new ParameterImpl("dest", (DataType)voidPtr, this.program, SourceType.ANALYSIS));
            GoFuncData funcData = this.goBinary.getFunctionByName("runtime.gcWriteBarrier");
            Function function = func = funcData != null ? funcData.getFunction() : null;
            if (func != null) {
                func.updateFunction(ccname, (Variable)retVal, params, Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
            }
            if (this.goBinary.getBuildInfo().getGOARCH(this.program).equals("amd64")) {
                Language lang = this.program.getLanguage();
                Register destReg = lang.getRegister("RDI");
                for (String regName : GoConstants.GCWRITE_BUFFERED_x86_64_Regs) {
                    Object gregName = regName.startsWith("R") ? regName : "R" + regName;
                    funcData = this.goBinary.getFunctionByName("runtime.gcWriteBarrier" + regName);
                    func = funcData != null ? funcData.getFunction() : null;
                    Register reg = lang.getRegister((String)gregName);
                    if (func == null || reg == null) continue;
                    params = List.of(new ParameterImpl("value", (DataType)voidPtr, reg, this.program, SourceType.ANALYSIS), new ParameterImpl("dest", (DataType)voidPtr, destReg, this.program, SourceType.ANALYSIS));
                    func.updateFunction(ccname, (Variable)retVal, params, Function.FunctionUpdateType.CUSTOM_STORAGE, true, SourceType.ANALYSIS);
                }
            }
        }
        catch (DuplicateNameException | InvalidInputException e) {
            Msg.error((Object)this, (Object)"Failed to update gcwrite function", (Throwable)e);
        }
    }

    private void fixDuffFunctions() {
        FunctionManager funcMgr = this.program.getFunctionManager();
        GoRegisterInfo regInfo = this.goBinary.getRegInfo();
        Pointer voidPtr = this.program.getDataTypeManager().getPointer((DataType)VoidDataType.dataType);
        GoFuncData duffzeroFuncdata = this.goBinary.getFunctionByName("runtime.duffzero");
        Function duffzeroFunc = duffzeroFuncdata != null ? funcMgr.getFunctionAt(duffzeroFuncdata.getFuncAddress()) : null;
        List<Variable> duffzeroParams = regInfo.getDuffzeroParams(this.program);
        if (duffzeroFunc != null && !duffzeroParams.isEmpty()) {
            Function duffcopyFunc;
            try {
                ReturnParameterImpl voidRet = new ReturnParameterImpl((DataType)VoidDataType.dataType, VariableStorage.VOID_STORAGE, this.program);
                duffzeroFunc.updateFunction("duffzero", (Variable)voidRet, duffzeroParams, Function.FunctionUpdateType.CUSTOM_STORAGE, true, SourceType.ANALYSIS);
                this.markupSession.appendComment(duffzeroFunc, null, "Golang special function: duffzero");
                this.aam.schedule(new FixupDuffAlternateEntryPointsBackgroundCommand(duffzeroFuncdata, duffzeroFunc), PROP_RTTI_PRIORITY.priority());
            }
            catch (DuplicateNameException | InvalidInputException e) {
                Msg.error((Object)this, (Object)"Failed to update main duffzero function", (Throwable)e);
            }
            GoFuncData duffcopyFuncdata = this.goBinary.getFunctionByName("runtime.duffcopy");
            Function function = duffcopyFunc = duffcopyFuncdata != null ? funcMgr.getFunctionAt(duffcopyFuncdata.getFuncAddress()) : null;
            if (duffcopyFuncdata != null && this.goBinary.hasCallingConvention("duffcopy")) {
                try {
                    List<ParameterImpl> params = List.of(new ParameterImpl("dest", (DataType)voidPtr, this.program), new ParameterImpl("src", (DataType)voidPtr, this.program));
                    duffcopyFunc.updateFunction("duffcopy", (Variable)new ReturnParameterImpl((DataType)VoidDataType.dataType, this.program), params, Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS, true, SourceType.ANALYSIS);
                    this.markupSession.appendComment(duffcopyFunc, null, "Golang special function: duffcopy");
                    this.aam.schedule(new FixupDuffAlternateEntryPointsBackgroundCommand(duffcopyFuncdata, duffcopyFunc), AnalysisPriority.FUNCTION_ANALYSIS.after().priority());
                }
                catch (DuplicateNameException | InvalidInputException e) {
                    Msg.error((Object)this, (Object)"Failed to update main duffcopy function", (Throwable)e);
                }
            }
        }
    }

    private Set<String> readNoReturnFuncNames() {
        HashSet<String> noreturnFuncnames = new HashSet<String>();
        try {
            for (ResourceFile file : NonReturningFunctionNames.findDataFiles(this.program)) {
                FileUtilities.getLines((ResourceFile)file).stream().map(String::trim).filter(s -> !s.isBlank() && !s.startsWith("#")).forEach(noreturnFuncnames::add);
            }
        }
        catch (XmlParseException | IOException e) {
            Msg.error((Object)this, (Object)"Failed to read Go noreturn func data file", (Throwable)e);
        }
        return noreturnFuncnames;
    }

    private Address createFakeContextMemory(long len) {
        long offset_from_eom = 0x100000L;
        Address max = this.program.getAddressFactory().getDefaultAddressSpace().getMaxAddress();
        Address mbStart = max.subtract(offset_from_eom + len - 1L);
        MemoryBlock newMB = MemoryBlockUtils.createUninitializedBlock(this.program, false, "ARTIFICAL_GOLANG_CONTEXT", mbStart, len, "Artifical memory block created to hold Go context data types", null, true, true, false, null);
        newMB.setArtificial(true);
        return newMB.getStart();
    }

    private void setupProgramContext() throws IOException {
        Address contextMemoryAddr;
        GoRegisterInfo goRegInfo = this.goBinary.getRegInfo();
        if (goRegInfo.getZeroRegister() != null && !goRegInfo.isZeroRegisterIsBuiltin()) {
            try {
                for (AddressRange textRange : this.goBinary.getTextAddresses().getAddressRanges()) {
                    this.program.getProgramContext().setValue(goRegInfo.getZeroRegister(), textRange.getMinAddress(), textRange.getMaxAddress(), BigInteger.ZERO);
                }
            }
            catch (ContextChangeException e) {
                Msg.error((Object)this, (Object)"Unexpected Error", (Throwable)e);
            }
        }
        int alignment = this.goBinary.getPtrSize();
        long sizeNeeded = 0L;
        Symbol zerobase = this.goBinary.getGoSymbol("runtime.zerobase");
        long zerobaseSymbol = sizeNeeded;
        long gStructOffset = sizeNeeded += zerobase == null ? NumericUtilities.getUnsignedAlignedValue((long)1L, (long)alignment) : 0L;
        Structure gStruct = this.goTypes.findDataType("runtime.g", Structure.class);
        long mStructOffset = sizeNeeded += gStruct != null ? NumericUtilities.getUnsignedAlignedValue((long)gStruct.getLength(), (long)alignment) : 0L;
        Structure mStruct = this.goTypes.findDataType("runtime.m", Structure.class);
        Address address = contextMemoryAddr = (sizeNeeded += mStruct != null ? NumericUtilities.getUnsignedAlignedValue((long)mStruct.getLength(), (long)alignment) : 0L) > 0L ? this.createFakeContextMemory(sizeNeeded) : null;
        if (zerobase == null) {
            this.markupSession.labelAddress(contextMemoryAddr.add(zerobaseSymbol), "ARTIFICIAL.runtime.zerobase");
        }
        if (gStruct != null) {
            Address gAddr = contextMemoryAddr.add(gStructOffset);
            this.markupSession.markupAddressIfUndefined(gAddr, (DataType)gStruct);
            this.markupSession.labelAddress(gAddr, "CURRENT_G");
            Register currentGoroutineReg = goRegInfo.getCurrentGoroutineRegister();
            if (currentGoroutineReg != null) {
                try {
                    for (AddressRange textRange : this.goBinary.getTextAddresses().getAddressRanges()) {
                        this.program.getProgramContext().setValue(currentGoroutineReg, textRange.getMinAddress(), textRange.getMaxAddress(), gAddr.getOffsetAsBigInteger());
                    }
                }
                catch (ContextChangeException e) {
                    Msg.error((Object)this, (Object)"Unexpected Error", (Throwable)e);
                }
            }
        }
        if (mStruct != null) {
            Address mAddr = contextMemoryAddr.add(mStructOffset);
            this.markupSession.markupAddressIfUndefined(mAddr, (DataType)mStruct);
        }
    }

    public static boolean isAlreadyAnalyzed(Program program) {
        Options options = program.getOptions("Program Information");
        return options.getBoolean(ANALYZED_FLAG_OPTION_NAME, false);
    }

    private static class GolangAnalyzerOptions {
        static final String OUTPUT_SOURCE_INFO_OPTIONNAME = "Output Source Info";
        static final String OUTPUT_SOURCE_INFO_DESC = "Add \"source_file_name:line_number\" information to functions.";
        boolean outputSourceInfo = true;
        static final String FIXUP_DUFF_FUNCS_OPTIONNAME = "Fixup Duff Functions";
        static final String FIXUP_DUFF_FUNCS_DESC = "Copies information from the runtime.duffzero and runtime.duffcopy functions to the alternate duff entry points that are discovered during later analysis.";
        boolean fixupDuffFunctions = true;
        static final String PROP_RTTI_OPTIONNAME = "Propagate RTTI";
        static final String PROP_RTTI_DESC = "Override the function signature of calls to some built-in Go allocator functions (runtime.newobject(), runtime.makeslice(), etc) that have a constant reference to a Go type record to have a return type of that specific Go type.";
        boolean propagateRtti = true;
        static final String FIXUP_GCWRITEBARRIER_OPTIONNAME = "Fixup gcWriteBarrier Functions";
        static final String FIXUP_GCWRITEBARRIER_FUNCS_DESC = "Fixup gcWriteBarrier functions (requires gcwrite calling convention defined for the program's arch)";
        boolean fixupGcWriteBarrierFunctions = true;
        static final String FIXUP_GCWRITEBARRIER_FLAG_OPTIONNAME = "Fixup gcWriteBarrier Flag";
        static final String FIXUP_GCWRITEBARRIER_FLAG_DESC = "Fixup global writeBarrier flag so decompiler can eliminate some code paths.\nThe un-initialized memory block the flag is located in will be split to enable\ninitializing the flag byte to 0.";
        public boolean fixupGcWriteBarrierFlag = true;

        private GolangAnalyzerOptions() {
        }

        void registerOptions(Options options, Program program) {
            options.registerOption(OUTPUT_SOURCE_INFO_OPTIONNAME, (Object)this.outputSourceInfo, null, OUTPUT_SOURCE_INFO_DESC);
            options.registerOption(FIXUP_DUFF_FUNCS_OPTIONNAME, (Object)this.fixupDuffFunctions, null, FIXUP_DUFF_FUNCS_DESC);
            options.registerOption(PROP_RTTI_OPTIONNAME, (Object)this.propagateRtti, null, PROP_RTTI_DESC);
            options.registerOption(FIXUP_GCWRITEBARRIER_OPTIONNAME, (Object)this.fixupGcWriteBarrierFunctions, null, FIXUP_GCWRITEBARRIER_FUNCS_DESC);
            options.registerOption(FIXUP_GCWRITEBARRIER_FLAG_OPTIONNAME, (Object)this.fixupGcWriteBarrierFlag, null, FIXUP_GCWRITEBARRIER_FLAG_DESC);
        }

        void optionsChanged(Options options, Program program) {
            this.outputSourceInfo = options.getBoolean(OUTPUT_SOURCE_INFO_OPTIONNAME, this.outputSourceInfo);
            this.fixupDuffFunctions = options.getBoolean(FIXUP_DUFF_FUNCS_OPTIONNAME, this.fixupDuffFunctions);
            this.propagateRtti = options.getBoolean(PROP_RTTI_OPTIONNAME, this.propagateRtti);
            this.fixupGcWriteBarrierFunctions = options.getBoolean(FIXUP_GCWRITEBARRIER_OPTIONNAME, this.fixupGcWriteBarrierFunctions);
            this.fixupGcWriteBarrierFlag = options.getBoolean(FIXUP_GCWRITEBARRIER_FLAG_OPTIONNAME, this.fixupGcWriteBarrierFlag);
        }
    }

    private static class PropagateRttiBackgroundCommand
    extends BackgroundCommand<Program> {
        private GoRttiMapper goBinary;
        private Program program;
        private MarkupSession markupSession;
        private int totalCallsiteCount;
        private int fixedCallsiteCount;
        private int unfixedCallsiteCount;
        private int callingFunctionCount;
        private Structure baseGoRttiTypeStruct;
        private List<RttiFuncInfo> rttiAllocFuncs = List.of(new RttiFuncInfo("runtime.newobject", 0, this::getReturnTypeForNewObjectFunc), new RttiFuncInfo("runtime.makeslice", 0, this::getReturnTypeForSliceFunc), new RttiFuncInfo("runtime.growslice", -1, this::getReturnTypeForSliceFunc), new RttiFuncInfo("runtime.makeslicecopy", 0, this::getReturnTypeForSliceFunc));

        public PropagateRttiBackgroundCommand(GoRttiMapper goBinary, MarkupSession markupSession) {
            super("Golang RTTI Propagation (deferred)", true, true, false);
            this.goBinary = goBinary;
            this.program = goBinary.getProgram();
            this.markupSession = markupSession;
        }

        public boolean applyTo(Program program, TaskMonitor monitor) {
            if (!this.goBinary.getRegInfo().hasAbiInternalParamRegisters()) {
                return true;
            }
            try {
                this.baseGoRttiTypeStruct = this.goBinary.getStructureDataType(GoBaseType.class);
                Set<Map.Entry<Function, List<CallSiteInfo>>> callsiteInfo = this.getInformationAboutCallsites(monitor);
                monitor.initialize((long)this.totalCallsiteCount, "Propagating Go RTTI from callsites");
                for (Map.Entry<Function, List<CallSiteInfo>> callsite : callsiteInfo) {
                    monitor.checkCancelled();
                    this.fixupRttiCallsitesInFunc(callsite.getKey(), callsite.getValue(), monitor);
                }
                Msg.info((Object)((Object)this), (Object)"Go RTTI callsite fixup info (total/updated/skipped): %d/%d/%d".formatted(this.totalCallsiteCount, this.fixedCallsiteCount, this.unfixedCallsiteCount));
                return true;
            }
            catch (CancelledException e) {
                return false;
            }
        }

        private void fixupRttiCallsitesInFunc(Function callingFunc, List<CallSiteInfo> callsites, TaskMonitor monitor) throws CancelledException {
            monitor.setMessage("Propagating Go RTTI from callsites in %s@%s".formatted(callingFunc.getName(), callingFunc.getEntryPoint()));
            GoTypeManager goTypes = this.goBinary.getGoTypes();
            ConstantPropagationContextEvaluator eval = new ConstantPropagationContextEvaluator(monitor, true);
            SymbolicPropogator symEval = new SymbolicPropogator(this.program, true);
            symEval.flowConstants(callingFunc.getEntryPoint(), callingFunc.getBody(), (ContextEvaluator)eval, true, monitor);
            monitor.setMessage("Propagating Go RTTI from callsites in %s@%s".formatted(callingFunc.getName(), callingFunc.getEntryPoint()));
            for (CallSiteInfo callsite : callsites) {
                monitor.increment();
                SymbolicPropogator.Value val = symEval.getRegisterValue(callsite.ref.getFromAddress(), callsite.register);
                if (val == null || val.isRegisterRelativeValue()) {
                    ++this.unfixedCallsiteCount;
                    continue;
                }
                long goTypeOffset = val.getValue();
                try {
                    DataType newReturnType;
                    GoType goType = goTypes.getType(goTypeOffset, true);
                    if (goType == null) {
                        goType = goTypes.getType(goTypeOffset);
                        this.markupSession.markup(goType, false);
                    }
                    if ((newReturnType = goType != null ? callsite.returnTypeMapper.apply(goType) : null) == null) continue;
                    FunctionDefinitionDataType signature = new FunctionDefinitionDataType(callsite.calledFunc, true);
                    signature.setReturnType(newReturnType);
                    HighFunctionDBUtil.writeOverride((Function)callsite.callingFunc, (Address)callsite.ref.getFromAddress(), (FunctionSignature)signature);
                    ++this.fixedCallsiteCount;
                }
                catch (InvalidInputException | IOException e) {
                    this.markupSession.logWarningAt(callsite.ref.getFromAddress(), "Failed to override with RTTI: " + e.getMessage());
                    ++this.unfixedCallsiteCount;
                }
            }
        }

        Set<Map.Entry<Function, List<CallSiteInfo>>> getInformationAboutCallsites(TaskMonitor monitor) {
            UnknownProgressWrappingTaskMonitor upwtm = new UnknownProgressWrappingTaskMonitor(monitor);
            upwtm.initialize(1L, "Finding callsites with Go RTTI");
            BiConsumer<RttiFuncInfo, Consumer> getReferencesToRttiFuncWithMonitor = (arg_0, arg_1) -> this.lambda$getInformationAboutCallsites$0((TaskMonitor)upwtm, arg_0, arg_1);
            Map<Function, List<CallSiteInfo>> result = this.rttiAllocFuncs.stream().mapMulti(getReferencesToRttiFuncWithMonitor).collect(Collectors.groupingBy(csi -> csi.callingFunc));
            this.callingFunctionCount = result.size();
            return result.entrySet();
        }

        private void getReferencesToRttiFunc(RttiFuncInfo rfi, Consumer<CallSiteInfo> consumer, TaskMonitor monitor) {
            FunctionManager funcMgr = this.program.getFunctionManager();
            ReferenceManager refMgr = this.program.getReferenceManager();
            Function func = rfi.funcName.getFunction(this.program);
            if (func != null) {
                List<Register> callRegs = this.getRegistersForTypeParameter(func, rfi.rttiParamIndex);
                if (callRegs == null || callRegs.size() != 1) {
                    return;
                }
                Register paramReg = callRegs.get(0);
                StreamSupport.stream(refMgr.getReferencesTo(func.getEntryPoint()).spliterator(), false).filter(ref -> !monitor.isCancelled() && ref != null && ref.getReferenceType().isCall()).map(ref -> new CallSiteInfo((Reference)ref, funcMgr.getFunctionContaining(ref.getFromAddress()), func, paramReg, rfi.returnTypeMapper)).forEach(consumer.andThen(_unused -> {
                    monitor.incrementProgress();
                    ++this.totalCallsiteCount;
                }));
            }
        }

        private DataType getReturnTypeForNewObjectFunc(GoType goType) {
            try {
                DataTypeManager dtm = this.goBinary.getDTM();
                DataType dt = this.goBinary.getGoTypes().getDataType(goType);
                return dtm.getPointer(dt);
            }
            catch (IOException e) {
                return null;
            }
        }

        private DataType getReturnTypeForSliceFunc(GoType goType) {
            try {
                GoTypeManager goTypes = this.goBinary.getGoTypes();
                GoType sliceGoType = goTypes.findGoType("[]" + goTypes.getTypeName(goType));
                DataType dt = sliceGoType != null ? goTypes.getDataType(sliceGoType) : null;
                return dt;
            }
            catch (IOException e) {
                return null;
            }
        }

        private List<Register> getRegistersForTypeParameter(Function func, int paramIndex) {
            GoParamStorageAllocator storageAllocator = this.goBinary.newStorageAllocator();
            Parameter[] params = func.getParameters();
            if (params.length == 0 && paramIndex == 0) {
                return storageAllocator.getRegistersFor(this.goBinary.getGoTypes().getVoidPtrDT());
            }
            for (int i = 0; i < params.length; ++i) {
                DataType paramDT = params[i].getDataType();
                List<Register> regs = storageAllocator.getRegistersFor(paramDT);
                if (i != paramIndex && (paramIndex != -1 || !this.isPtrToGoType(paramDT))) continue;
                return regs;
            }
            return List.of();
        }

        private boolean isPtrToGoType(DataType dt) {
            if (!(dt instanceof Pointer)) {
                return false;
            }
            Pointer ptrDT = (Pointer)dt;
            DataType elementDT = ptrDT.getDataType();
            if (elementDT instanceof TypeDef) {
                TypeDef tdDT = (TypeDef)elementDT;
                elementDT = tdDT.getBaseDataType();
            }
            return elementDT != null && elementDT.equals((Object)this.baseGoRttiTypeStruct);
        }

        private /* synthetic */ void lambda$getInformationAboutCallsites$0(TaskMonitor upwtm, RttiFuncInfo rfi, Consumer c) {
            this.getReferencesToRttiFunc(rfi, c, upwtm);
        }

        record RttiFuncInfo(GoSymbolName funcName, int rttiParamIndex, java.util.function.Function<GoType, DataType> returnTypeMapper) {
            public RttiFuncInfo(String funcName, int rttiParamIndex, java.util.function.Function<GoType, DataType> returnTypeMapper) {
                this(GoSymbolName.parse(funcName), rttiParamIndex, returnTypeMapper);
            }
        }

        record CallSiteInfo(Reference ref, Function callingFunc, Function calledFunc, Register register, java.util.function.Function<GoType, DataType> returnTypeMapper) {
        }
    }

    private static class FixClosureFuncArgsBackgroundCommand
    extends BackgroundCommand<Program> {
        private GoRttiMapper goBinary;
        private Program program;
        private ReferenceManager refMgr;
        private Register closureContextRegister;
        private GoTypeManager goTypes;
        private int closureTypeCount;
        private int methodWrapperTypeCount;
        private int closureFuncsFixed;
        private int methodWrapperFuncsFixed;

        public FixClosureFuncArgsBackgroundCommand(GoRttiMapper goBinary) {
            super("Golang closure func arg (deferred)", true, true, false);
            this.goBinary = goBinary;
            this.goTypes = goBinary.getGoTypes();
            this.program = goBinary.getProgram();
            this.refMgr = this.program.getReferenceManager();
            this.closureContextRegister = goBinary.getRegInfo().getClosureContextRegister();
        }

        public boolean applyTo(Program obj, TaskMonitor monitor) {
            for (GoType closureType : this.goTypes.getClosureTypes()) {
                if (monitor.isCancelled()) {
                    return false;
                }
                this.fixupFuncsWithClosureRefsToTypeStruct(closureType, false, monitor);
                ++this.closureTypeCount;
            }
            for (GoType closureType : this.goTypes.getMethodWrapperClosureTypes()) {
                if (monitor.isCancelled()) {
                    return false;
                }
                this.fixupFuncsWithClosureRefsToTypeStruct(closureType, true, monitor);
                ++this.methodWrapperTypeCount;
            }
            Msg.info((Object)((Object)this), (Object)"Go closure/method wrapper types found: %d/%d".formatted(this.closureTypeCount, this.methodWrapperTypeCount));
            Msg.info((Object)((Object)this), (Object)"Go closure/method wrapper funcs fixed: %d/%d".formatted(this.closureFuncsFixed, this.methodWrapperFuncsFixed));
            return true;
        }

        private void fixupFuncsWithClosureRefsToTypeStruct(GoType closureStructType, boolean isMethodWrapper, TaskMonitor monitor) {
            Address typStructAddr = closureStructType.getStructureContext().getStructureAddress();
            HashSet funcsProcessed = new HashSet();
            StreamSupport.stream(this.refMgr.getReferencesTo(typStructAddr).spliterator(), false).filter(ref -> !monitor.isCancelled() && ref != null && ref.getReferenceType().isData()).map(ref -> this.getNextClosureFuncRef(ref.getFromAddress(), isMethodWrapper, 50)).filter(funcData -> funcData != null && !funcsProcessed.contains(funcData.getFuncAddress())).forEach(funcData -> {
                Address addr = funcData.getFuncAddress();
                funcsProcessed.add(addr);
                Function func = this.program.getFunctionManager().getFunctionAt(addr);
                if (func != null) {
                    if (!this.isOverwriteableClosureFunc(func)) {
                        return;
                    }
                    if (isMethodWrapper) {
                        this.fixupMethodWrapperClosureFunc((GoFuncData)funcData, func, closureStructType);
                    } else {
                        this.fixupClosureFunc((GoFuncData)funcData, func, closureStructType);
                    }
                }
            });
        }

        private boolean isOverwriteableClosureFunc(Function func) {
            Parameter[] params = func.getParameters();
            return params.length == 0 || params.length == 1 && GoFunctionFixup.isClosureContext(params[0]);
        }

        private GoFuncData getNextClosureFuncRef(Address startAddr, boolean isMethodWrapper, int maxRange) {
            Address refAddr;
            if (!startAddr.isMemoryAddress()) {
                return null;
            }
            Address maxAddr = startAddr.add((long)maxRange);
            Iterator iterator = this.refMgr.getReferenceSourceIterator(startAddr, true).iterator();
            while (iterator.hasNext() && (refAddr = (Address)iterator.next()).compareTo((Object)maxAddr) <= 0) {
                for (Reference ref : this.refMgr.getReferencesFrom(refAddr)) {
                    GoSymbolName funcName;
                    if (!ref.getReferenceType().isData()) continue;
                    Address destAddr = ref.getToAddress();
                    GoFuncData funcData = this.goBinary.getFunctionData(destAddr);
                    GoSymbolName goSymbolName = funcName = funcData != null ? funcData.getSymbolName() : null;
                    if (funcName == null) continue;
                    GoSymbolNameType nameType = funcName.getNameType();
                    if (isMethodWrapper && nameType == GoSymbolNameType.METHOD_WRAPPER) {
                        return funcData;
                    }
                    if (isMethodWrapper || nameType == null || !nameType.isClosure()) continue;
                    return funcData;
                }
            }
            return null;
        }

        private void fixupClosureFunc(GoFuncData funcData, Function func, GoType closureStructType) {
            try {
                DataType closureStructDT = this.goTypes.getDataType(closureStructType);
                List<ParameterImpl> closureParams = List.of(new ParameterImpl(".context", (DataType)this.goBinary.getDTM().getPointer(closureStructDT), this.closureContextRegister, this.program, SourceType.ANALYSIS));
                func.updateFunction(null, null, closureParams, Function.FunctionUpdateType.CUSTOM_STORAGE, true, SourceType.ANALYSIS);
                ++this.closureFuncsFixed;
            }
            catch (DuplicateNameException | InvalidInputException | IOException e) {
                Msg.error((Object)((Object)this), (Object)"Failed to update closure func signature %s@%s".formatted(func.getName(), func.getEntryPoint()), (Throwable)e);
            }
        }

        private void fixupMethodWrapperClosureFunc(GoFuncData funcData, Function func, GoType closureStructType) {
            Function methodFunc;
            String methodName = func.getName();
            GoFuncData methodFuncData = this.goBinary.getFunctionByName(methodName = methodName.substring(0, methodName.length() - "-fm".length()));
            if (methodFuncData == null) {
                try {
                    DataType closureStructDT = this.goTypes.getDataType(closureStructType);
                    List<ParameterImpl> closureParams = List.of(new ParameterImpl(".context", (DataType)this.goBinary.getDTM().getPointer(closureStructDT), this.closureContextRegister, this.program, SourceType.ANALYSIS));
                    func.updateFunction(null, null, closureParams, Function.FunctionUpdateType.CUSTOM_STORAGE, true, SourceType.ANALYSIS);
                    ++this.methodWrapperFuncsFixed;
                    return;
                }
                catch (DuplicateNameException | InvalidInputException | IOException e) {
                    Msg.error((Object)((Object)this), (Object)"Failed to update closure func signature %s@%s".formatted(func.getName(), func.getEntryPoint()), (Throwable)e);
                }
            }
            if ((methodFunc = this.program.getFunctionManager().getFunctionAt(methodFuncData.getFuncAddress())) == null) {
                return;
            }
            try {
                DataType closureStructDT = this.goTypes.getDataType(closureStructType);
                Parameter methodReturn = methodFunc.getReturn();
                Parameter[] methodParams = methodFunc.getParameters();
                methodParams[0] = new ParameterImpl(".context", (DataType)this.goBinary.getDTM().getPointer(closureStructDT), this.closureContextRegister, this.program, SourceType.ANALYSIS);
                func.updateFunction(null, (Variable)methodReturn, Function.FunctionUpdateType.CUSTOM_STORAGE, true, SourceType.ANALYSIS, (Variable[])methodParams);
                ++this.methodWrapperFuncsFixed;
            }
            catch (DuplicateNameException | InvalidInputException | IOException e) {
                Msg.error((Object)((Object)this), (Object)"Failed to update closure func signature %s@%s".formatted(func.getName(), func.getEntryPoint()), (Throwable)e);
            }
        }
    }

    private static class FixGcWriteBarrierFlagBackgroundCommand
    extends BackgroundCommand<Program> {
        private GoRttiMapper goBinary;
        private Program program;
        private ReferenceManager refMgr;
        private MarkupSession markupSession;
        private AddressSetView searchAddrs;
        private static final int MAX_REFSEARCH_BYTES = 50;
        private static final Pattern GCWRITEBARRIER_FUNCNAMES = Pattern.compile("runtime\\.gcWriteBarrier.*");

        public FixGcWriteBarrierFlagBackgroundCommand(GoRttiMapper goBinary, MarkupSession markupSession) {
            super("Golang writeBarrier flag fixup (deferred)", true, true, false);
            this.goBinary = goBinary;
            this.markupSession = markupSession;
            this.program = goBinary.getProgram();
            this.refMgr = this.program.getReferenceManager();
            this.searchAddrs = goBinary.getUninitializedNoPtrDataRange();
        }

        public boolean applyTo(Program obj, TaskMonitor monitor) {
            Memory memory = this.program.getMemory();
            Address flagAddr = this.getGcWriteBarrierFlagAddr();
            if (flagAddr == null) {
                Msg.warn((Object)((Object)this), (Object)"Could not find runtime.writeBarrier, unable to fixup");
                return true;
            }
            MemoryBlock memBlk = memory.getBlock(flagAddr);
            if (!memBlk.isInitialized()) {
                try {
                    memory.split(memBlk, flagAddr);
                    memBlk = memory.getBlock(flagAddr);
                    Address afterFlag = flagAddr.add(4L);
                    memory.split(memBlk, afterFlag);
                    memory.convertToInitialized(memBlk, (byte)0);
                    memBlk.setName(memBlk.getName().replaceFirst("\\.split$", ".writeBarrierFlag"));
                    memBlk = memory.getBlock(afterFlag);
                    memBlk.setName(memBlk.getName().replaceFirst("(\\.split)+$", ".part2"));
                }
                catch (LockException | MemoryBlockException | NotFoundException e) {
                    Msg.error((Object)((Object)this), (Object)"Failed to fixup runtime.writeBarrier flag", (Throwable)e);
                }
            } else {
                try {
                    memory.setBytes(flagAddr, new byte[4]);
                }
                catch (MemoryAccessException e) {
                    Msg.error((Object)((Object)this), (Object)"Failed to set runtime.writeBarrier flag bytes to 0");
                }
            }
            try {
                Data flagData = DataUtilities.createData((Program)this.program, (Address)flagAddr, (DataType)AbstractIntegerDataType.getUnsignedDataType((int)4, null), (int)4, (DataUtilities.ClearDataMode)DataUtilities.ClearDataMode.CLEAR_ALL_CONFLICT_DATA);
                MutabilitySettingsDefinition.DEF.setChoice((Settings)flagData, 2);
                this.markupSession.labelAddress(flagAddr, "runtime.writeBarrier.discovered");
            }
            catch (CodeUnitInsertionException | IOException e) {
                Msg.error((Object)((Object)this), (Object)"Failed to markup runtime.writeBarrier flag", (Throwable)e);
            }
            return true;
        }

        private Address getGcWriteBarrierFlagAddr() {
            Optional mostHits;
            Symbol writeBarrierSym = this.goBinary.getGoSymbol("runtime.writeBarrier");
            if (writeBarrierSym != null) {
                return writeBarrierSym.getAddress();
            }
            HashMap<Address, Integer> candidateFlagAddrs = new HashMap<Address, Integer>();
            for (Address wbFunc : this.getGcWriteBarrierFuncAddrs()) {
                for (Reference callSiteRef : this.refMgr.getReferencesTo(wbFunc)) {
                    if (!callSiteRef.getReferenceType().isCall()) continue;
                    this.getGcWriteBarrierFlagAddrGuesses(callSiteRef.getFromAddress(), candidateFlagAddrs);
                }
            }
            if (candidateFlagAddrs.size() > 1) {
                Msg.debug((Object)((Object)this), (Object)("runtime.writeBarrier flag candidates: " + String.valueOf(candidateFlagAddrs)));
            }
            return (mostHits = candidateFlagAddrs.entrySet().stream().sorted((e1, e2) -> Integer.compare((Integer)e2.getValue(), (Integer)e1.getValue())).findFirst()).isPresent() ? (Address)((Map.Entry)mostHits.get()).getKey() : null;
        }

        private void getGcWriteBarrierFlagAddrGuesses(Address wbFuncCallSite, Map<Address, Integer> candidateFlagAddrs) {
            Address refAddr;
            Address minAddr = wbFuncCallSite.subtract(50L);
            Iterator iterator = this.refMgr.getReferenceSourceIterator(wbFuncCallSite, false).iterator();
            while (iterator.hasNext() && (refAddr = (Address)iterator.next()).compareTo((Object)minAddr) >= 0) {
                for (Reference ref : this.refMgr.getReferencesFrom(refAddr)) {
                    if (!ref.getReferenceType().isData() || !ref.getReferenceType().isRead() || !ref.isMemoryReference() || !this.searchAddrs.contains(ref.getToAddress()) || !this.hasTrailingConditionalRef(refAddr, wbFuncCallSite)) continue;
                    candidateFlagAddrs.compute(ref.getToAddress(), (k, v) -> v == null ? 1 : v + 1);
                }
            }
        }

        private boolean hasTrailingConditionalRef(Address addr, Address maxAddr) {
            Address refAddr;
            Iterator iterator = this.refMgr.getReferenceSourceIterator(addr, true).iterator();
            while (iterator.hasNext() && (refAddr = (Address)iterator.next()).compareTo((Object)maxAddr) <= 0) {
                for (Reference ref : this.refMgr.getReferencesFrom(refAddr)) {
                    if (!ref.getReferenceType().isConditional() || !ref.getReferenceType().isJump()) continue;
                    return true;
                }
            }
            return false;
        }

        private List<Address> getGcWriteBarrierFuncAddrs() {
            return this.goBinary.getFunctionsByNamePattern(GCWRITEBARRIER_FUNCNAMES).stream().map(GoFuncData::getFuncAddress).toList();
        }
    }

    private static class FixupDuffAlternateEntryPointsBackgroundCommand
    extends BackgroundCommand<Program> {
        private Function duffFunc;
        private GoFuncData funcData;

        public FixupDuffAlternateEntryPointsBackgroundCommand(GoFuncData funcData, Function duffFunc) {
            super("Golang duffzero/duffcopy fixups (deferred)", true, true, false);
            this.funcData = funcData;
            this.duffFunc = duffFunc;
        }

        public boolean applyTo(Program program, TaskMonitor monitor) {
            if (!this.duffFunc.getProgram().equals((Object)program)) {
                throw new AssertionError();
            }
            String ccName = this.duffFunc.getCallingConventionName();
            Namespace funcNS = this.duffFunc.getParentNamespace();
            AddressSet funcBody = new AddressSet(this.funcData.getBody());
            String duffComment = program.getListing().getCodeUnitAt(this.duffFunc.getEntryPoint()).getComment(CommentType.PLATE);
            monitor.setMessage("Fixing alternate duffzero/duffcopy entry points");
            FunctionIterator funcIt = program.getFunctionManager().getFunctions((AddressSetView)funcBody, true);
            while (funcIt.hasNext()) {
                Function func = (Function)funcIt.next();
                if (!FunctionUtility.isDefaultFunctionName(func)) continue;
                try {
                    func.setName(this.duffFunc.getName() + "_" + String.valueOf(func.getEntryPoint()), SourceType.ANALYSIS);
                    func.setParentNamespace(funcNS);
                    Function.FunctionUpdateType fut = this.duffFunc.hasCustomVariableStorage() ? Function.FunctionUpdateType.CUSTOM_STORAGE : Function.FunctionUpdateType.DYNAMIC_STORAGE_ALL_PARAMS;
                    func.updateFunction(ccName, (Variable)this.duffFunc.getReturn(), Arrays.asList(this.duffFunc.getParameters()), fut, true, SourceType.ANALYSIS);
                    if (duffComment == null || duffComment.isBlank()) continue;
                    new SetCommentCmd(func.getEntryPoint(), CommentType.PLATE, duffComment).applyTo(program);
                }
                catch (CircularDependencyException | DuplicateNameException | InvalidInputException e) {
                    Msg.error(GolangSymbolAnalyzer.class, (Object)"Error updating duff functions", (Throwable)e);
                }
            }
            return true;
        }
    }
}

